import { defineStore } from 'pinia';
import api from '../services/api';

export const useAuthStore = defineStore('auth', {
  state: () => ({
    user: JSON.parse(localStorage.getItem('user')) || null,
    token: localStorage.getItem('token') || null,
    restaurantName: localStorage.getItem('restaurantName') || null,
  }),

  getters: {
    isAuthenticated: (state) => !!state.token,
    userRole: (state) => state.user?.roles?.[0]?.name || null,
    isManager: (state) => state.user?.roles?.[0]?.name === 'manager',
    isAdmin: (state) => state.user?.roles?.[0]?.name === 'admin',
    showRestaurantName: (state) => ['manager', 'admin'].includes(state.user?.roles?.[0]?.name),
  },

  actions: {
    async register(data) {
      const response = await api.post('/register', data);
      this.token = response.data.token;
      this.user = response.data.user;
      localStorage.setItem('token', this.token);
      localStorage.setItem('user', JSON.stringify(this.user));
    },

    async login(credentials) {
      const response = await api.post('/login', credentials);
      this.token = response.data.token;
      this.user = response.data.user;
      localStorage.setItem('token', this.token);
      localStorage.setItem('user', JSON.stringify(this.user));
      
      if (this.showRestaurantName) {
        await this.loadRestaurantName();
      }
    },

    async logout() {
      try {
        await api.post('/logout');
      } catch (error) {
        console.error('Logout error:', error);
      }
      this.token = null;
      this.user = null;
      this.restaurantName = null;
      localStorage.removeItem('token');
      localStorage.removeItem('user');
      localStorage.removeItem('restaurantName');
    },

    async checkAuth() {
      if (this.token) {
        try {
          const response = await api.get('/user');
          this.user = response.data;
          localStorage.setItem('user', JSON.stringify(this.user));
          
          if (this.showRestaurantName && !this.restaurantName) {
            await this.loadRestaurantName();
          }
        } catch (error) {
          this.token = null;
          this.user = null;
          this.restaurantName = null;
          localStorage.removeItem('token');
          localStorage.removeItem('user');
          localStorage.removeItem('restaurantName');
        }
      }
    },
    
    async loadRestaurantName() {
      try {
        const response = await api.get('/manager/restaurant');
        this.restaurantName = response.data.data.name;
        localStorage.setItem('restaurantName', this.restaurantName);
      } catch (error) {
        console.error('Failed to load restaurant name:', error);
      }
    },

    async forgotPassword(identifier) {
      const response = await api.post('/forgot-password', { identifier });
      return response.data;
    },

    async resetPassword(data) {
      const response = await api.post('/reset-password', data);
      return response.data;
    },
  },
});
