<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;

class RolePermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Reset cached roles and permissions
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // Create permissions for admin modules
        $permissions = [
            // Dashboard Analytics
            'view-dashboard-analytics',
            'view-revenue-charts',
            'view-order-charts',
            'view-user-charts',
            'view-manager-dashboard',
            'view-delivery-dashboard',
            
            // Restaurant Management
            'view-restaurants',
            'create-restaurants',
            'update-restaurants',
            'delete-restaurants',
            'approve-restaurants',
            'suspend-restaurants',
            'view-restaurant-analytics',
            'update-restaurant-commission',
            
            // Product Management
            'view-products',
            'create-products',
            'update-products',
            'delete-products',
            'toggle-product-availability',
            'view-product-analytics',
            'export-products',
            
            // Category Management
            'view-categories',
            'create-categories',
            'update-categories',
            'delete-categories',
            'toggle-category-status',
            'assign-categories-to-restaurants',
            'reorder-categories',
            
            // Delivery Boy Management
            'view-delivery-boys',
            'create-delivery-boys',
            'update-delivery-boys',
            'delete-delivery-boys',
            'approve-delivery-boys',
            'suspend-delivery-boys',
            'view-delivery-boy-analytics',
            'update-delivery-boy-commission',
            'process-delivery-boy-payouts',
            
            // Order Management
            'view-orders',
            'update-order-status',
            'cancel-orders',
            'refund-orders',
            'assign-delivery-boys-to-orders',
            'change-delivery-boys-in-orders',
            'view-order-timeline',
            'export-orders',
            
            // Payment & Commission Management
            'view-payments',
            'view-commissions',
            'manage-commission-rules',
            'view-payouts',
            'create-payouts',
            'process-payouts',
            'cancel-payouts',
            'view-wallet-transactions',
            'view-financial-reports',
            
            // User Management
            'view-users',
            'create-users',
            'update-users',
            'delete-users',
            'block-users',
            'unblock-users',
            'manage-user-wallets',
            'view-user-analytics',
            'export-users',
            
            // Settings Management
            'view-settings',
            'update-settings',
            'create-custom-settings',
            'delete-custom-settings',
            'manage-system-settings',
            'backup-settings',
            'restore-settings',
            'clear-cache',
            'view-system-info',
            'manage-commission-settings'
        ];

        // Create all permissions
        foreach ($permissions as $permission) {
            Permission::updateOrCreate(['name' => $permission, 'guard_name' => 'sanctum']);
        }

        // Create roles and assign permissions
        
        // Super Admin Role (has all permissions)
        $superAdmin = Role::updateOrCreate(['name' => 'super-admin', 'guard_name' => 'sanctum']);
        $superAdmin->givePermissionTo(Permission::all());

        // Admin Role (Shop/Restaurant Owner)
        $admin = Role::updateOrCreate(['name' => 'admin', 'guard_name' => 'sanctum']);
        $admin->givePermissionTo([
            'view-dashboard-analytics',
            'view-restaurants',
            'update-restaurants',
            'view-restaurant-analytics',
            'view-products',
            'create-products',
            'update-products',
            'delete-products',
            'toggle-product-availability',
            'view-product-analytics',
            'view-orders',
            'update-order-status',
            'cancel-orders',
            'view-order-timeline',
            'export-orders',
            'view-payments',
            'view-commissions',
            'view-users',
            'create-users',
            'update-users',
            'delete-users'
        ]);

        // Manager Role
        $manager = Role::updateOrCreate(['name' => 'manager', 'guard_name' => 'sanctum']);
        $manager->givePermissionTo([
            'view-manager-dashboard',
            'view-restaurants',
            'create-restaurants',
            'update-restaurants',
            'view-restaurant-analytics',
            'view-products',
            'create-products',
            'update-products',
            'toggle-product-availability',
            'view-product-analytics',
            'view-categories',
            'create-categories',
            'update-categories',
            'toggle-category-status',
            'assign-categories-to-restaurants',
            'view-orders',
            'update-order-status',
            'view-payments',
            'view-commissions'
        ]);

        // Cashier Role
        $cashier = Role::updateOrCreate(['name' => 'cashier', 'guard_name' => 'sanctum']);
        $cashier->givePermissionTo([
            'view-orders',
            'update-order-status',
            'view-payments'
        ]);

        // Delivery Boy Role
        $deliveryBoy = Role::updateOrCreate(['name' => 'delivery-boy', 'guard_name' => 'sanctum']);
        $deliveryBoy->givePermissionTo([
            'view-delivery-dashboard',
            'view-delivery-boy-analytics'
        ]);

        // Customer Role
        $customer = Role::updateOrCreate(['name' => 'customer', 'guard_name' => 'sanctum']);
        $customer->givePermissionTo([
            // Basic customer permissions are handled through ownership policies
        ]);

        // Create default admin user if not exists
        if (!User::where('email', 'admin@fooddelivery.com')->exists()) {
            $adminUser = User::create([
                'name' => 'Super Admin',
                'email' => 'admin@fooddelivery.com',
                'password' => bcrypt('admin123'),
                'phone' => '+1234567890',
                'address' => '123 Admin Street, Admin City',
                'email_verified_at' => now()
            ]);
            
            $adminUser->assignRole($superAdmin);
        }

        // Create sample admin user
        if (!User::where('email', 'admin2@fooddelivery.com')->exists()) {
            $adminUser2 = User::create([
                'name' => 'Admin User',
                'email' => 'admin2@fooddelivery.com',
                'password' => bcrypt('admin123'),
                'phone' => '+1234567891',
                'address' => '456 Admin Avenue, Admin City',
                'email_verified_at' => now()
            ]);
            
            $adminUser2->assignRole($admin);
        }

        // Create sample manager
        if (!User::where('email', 'manager@fooddelivery.com')->exists()) {
            $managerUser = User::create([
                'name' => 'Restaurant Manager',
                'email' => 'manager@fooddelivery.com',
                'password' => bcrypt('manager123'),
                'phone' => '+1234567892',
                'address' => '789 Manager Road, Manager City',
                'email_verified_at' => now()
            ]);
            
            $managerUser->assignRole($manager);
        }

        // Create sample delivery boy
        if (!User::where('email', 'delivery@fooddelivery.com')->exists()) {
            $deliveryUser = User::create([
                'name' => 'Delivery Boy',
                'email' => 'delivery@fooddelivery.com',
                'password' => bcrypt('delivery123'),
                'phone' => '+1234567893',
                'address' => '321 Delivery Lane, Delivery City',
                'email_verified_at' => now()
            ]);
            
            $deliveryUser->assignRole($deliveryBoy);
        }

        // Create sample customer
        if (!User::where('email', 'customer@fooddelivery.com')->exists()) {
            $customerUser = User::create([
                'name' => 'John Customer',
                'email' => 'customer@fooddelivery.com',
                'password' => bcrypt('customer123'),
                'phone' => '+1234567894',
                'address' => '456 Customer Street, Colombo',
                'email_verified_at' => now()
            ]);
            
            $customerUser->assignRole($customer);
        }
    }
}