<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use App\Models\Product;
use App\Models\Category;
use App\Models\Restaurant;
use App\Models\User;

class ProductSeeder extends Seeder
{
    public function run(): void
    {
        // Get first manager user and their restaurant
        $manager = User::role('manager')->first();
        if (!$manager || !$manager->restaurant) {
            $this->command->warn('No manager with restaurant found. Please create a manager first.');
            return;
        }

        $restaurant = $manager->restaurant;

        // Get or create categories
        $categories = [
            'Burgers' => Category::firstOrCreate(['name' => 'Burgers']),
            'Pizza' => Category::firstOrCreate(['name' => 'Pizza']),
            'Beverages' => Category::firstOrCreate(['name' => 'Beverages']),
            'Desserts' => Category::firstOrCreate(['name' => 'Desserts']),
        ];

        // Create products
        $products = [
            [
                'name' => 'Classic Beef Burger',
                'description' => 'Juicy beef patty with lettuce, tomato, and special sauce',
                'category_id' => $categories['Burgers']->id,
                'price' => 850.00,
                'discount' => 10,
                'preparation_time' => 15,
                'variations' => [
                    ['name' => 'Size', 'value' => 'Regular', 'price' => 0],
                    ['name' => 'Size', 'value' => 'Large', 'price' => 200],
                ],
                'addons' => [
                    ['name' => 'Extra Cheese', 'price' => 100],
                    ['name' => 'Bacon', 'price' => 150],
                ]
            ],
            [
                'name' => 'Chicken Burger',
                'description' => 'Crispy chicken fillet with mayo and lettuce',
                'category_id' => $categories['Burgers']->id,
                'price' => 750.00,
                'discount' => 0,
                'preparation_time' => 12,
                'variations' => [
                    ['name' => 'Spice Level', 'value' => 'Mild', 'price' => 0],
                    ['name' => 'Spice Level', 'value' => 'Hot', 'price' => 50],
                ],
                'addons' => [
                    ['name' => 'Extra Sauce', 'price' => 50],
                ]
            ],
            [
                'name' => 'Margherita Pizza',
                'description' => 'Classic pizza with tomato sauce, mozzarella, and basil',
                'category_id' => $categories['Pizza']->id,
                'price' => 1200.00,
                'discount' => 15,
                'preparation_time' => 20,
                'variations' => [
                    ['name' => 'Size', 'value' => 'Small', 'price' => 0],
                    ['name' => 'Size', 'value' => 'Medium', 'price' => 300],
                    ['name' => 'Size', 'value' => 'Large', 'price' => 600],
                ],
                'addons' => [
                    ['name' => 'Extra Cheese', 'price' => 150],
                    ['name' => 'Olives', 'price' => 100],
                    ['name' => 'Mushrooms', 'price' => 120],
                ]
            ],
            [
                'name' => 'Pepperoni Pizza',
                'description' => 'Loaded with pepperoni and mozzarella cheese',
                'category_id' => $categories['Pizza']->id,
                'price' => 1500.00,
                'discount' => 0,
                'preparation_time' => 20,
                'variations' => [
                    ['name' => 'Size', 'value' => 'Medium', 'price' => 0],
                    ['name' => 'Size', 'value' => 'Large', 'price' => 400],
                ],
                'addons' => [
                    ['name' => 'Extra Pepperoni', 'price' => 200],
                ]
            ],
            [
                'name' => 'Coca Cola',
                'description' => 'Refreshing cold beverage',
                'category_id' => $categories['Beverages']->id,
                'price' => 150.00,
                'discount' => 0,
                'preparation_time' => 2,
                'variations' => [
                    ['name' => 'Size', 'value' => '330ml', 'price' => 0],
                    ['name' => 'Size', 'value' => '500ml', 'price' => 50],
                    ['name' => 'Size', 'value' => '1L', 'price' => 100],
                ],
                'addons' => []
            ],
            [
                'name' => 'Chocolate Brownie',
                'description' => 'Rich chocolate brownie with vanilla ice cream',
                'category_id' => $categories['Desserts']->id,
                'price' => 450.00,
                'discount' => 20,
                'preparation_time' => 8,
                'variations' => [],
                'addons' => [
                    ['name' => 'Extra Ice Cream', 'price' => 100],
                    ['name' => 'Chocolate Sauce', 'price' => 50],
                ]
            ],
        ];

        foreach ($products as $productData) {
            $variations = $productData['variations'];
            $addons = $productData['addons'];
            unset($productData['variations'], $productData['addons']);

            $product = Product::create([
                'restaurant_id' => $restaurant->id,
                'category_id' => $productData['category_id'],
                'name' => $productData['name'],
                'description' => $productData['description'],
                'price' => $productData['price'],
                'discount' => $productData['discount'],
                'preparation_time' => $productData['preparation_time'],
                'is_available' => true,
            ]);

            // Also add to restaurant_products pivot table
            DB::table('restaurant_products')->insert([
                'restaurant_id' => $restaurant->id,
                'product_id' => $product->id,
                'price' => $productData['price'],
                'offer_price' => $productData['discount'] > 0 ? $productData['price'] * (1 - $productData['discount']/100) : null,
                'is_available' => true,
                'preparation_time' => $productData['preparation_time'],
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Add variations
            foreach ($variations as $variation) {
                $product->variations()->create($variation);
            }

            // Add addons
            foreach ($addons as $addon) {
                $product->addons()->create($addon);
            }
        }

        $this->command->info('Products seeded successfully!');
    }
}
