<?php

namespace Database\Seeders;

use App\Models\Product;
use App\Models\Restaurant;
use App\Models\Category;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class ProductRestaurantSeeder extends Seeder
{
    public function run(): void
    {
        // Create 10 restaurants
        $restaurants = [];
        for ($i = 1; $i <= 10; $i++) {
            $email = "restaurant$i@test.com";
            
            // Check if user already exists
            $user = User::where('email', $email)->first();
            
            if (!$user) {
                $user = User::create([
                    'name' => "Restaurant Owner $i",
                    'email' => $email,
                    'password' => bcrypt('password'),
                    'phone' => '077' . str_pad($i, 7, '0', STR_PAD_LEFT),
                ]);
                
                $user->assignRole('manager');
            }
            
            // Check if restaurant already exists
            $restaurant = Restaurant::where('user_id', $user->id)->first();
            
            if (!$restaurant) {
                $restaurant = Restaurant::create([
                    'user_id' => $user->id,
                    'name' => "Restaurant $i",
                    'description' => "Description for Restaurant $i",
                    'phone' => '077' . str_pad($i, 7, '0', STR_PAD_LEFT),
                    'email' => $email,
                    'address' => "Address $i, Colombo",
                    'latitude' => 6.9271 + ($i * 0.01),
                    'longitude' => 79.8612 + ($i * 0.01),
                    'status' => 'approved',
                ]);
                
                $user->update(['restaurant_id' => $restaurant->id]);
            }
            
            $restaurants[] = $restaurant;
        }

        // Get categories
        $categories = Category::all();
        if ($categories->isEmpty()) {
            $categories = collect([
                Category::create(['name' => 'Rice & Curry']),
                Category::create(['name' => 'Kottu']),
                Category::create(['name' => 'Fried Rice']),
                Category::create(['name' => 'Noodles']),
                Category::create(['name' => 'Short Eats']),
            ]);
        }

        // Create 100 products
        $products = [];
        $productNames = [
            'Chicken Kottu', 'Egg Kottu', 'Vegetable Kottu', 'Cheese Kottu', 'Dolphin Kottu',
            'Chicken Fried Rice', 'Egg Fried Rice', 'Seafood Fried Rice', 'Mixed Fried Rice', 'Vegetable Fried Rice',
            'Chicken Noodles', 'Egg Noodles', 'Seafood Noodles', 'Vegetable Noodles', 'Mixed Noodles',
            'Rice & Curry', 'Fish Curry', 'Chicken Curry', 'Dhal Curry', 'Vegetable Curry',
            'Chicken Submarine', 'Egg Submarine', 'Fish Submarine', 'Cheese Submarine', 'Vegetable Submarine',
            'Chicken Roll', 'Egg Roll', 'Fish Roll', 'Vegetable Roll', 'Cheese Roll',
            'Chicken Burger', 'Beef Burger', 'Fish Burger', 'Cheese Burger', 'Vegetable Burger',
            'Chicken Pizza', 'Cheese Pizza', 'Vegetable Pizza', 'Seafood Pizza', 'BBQ Pizza',
            'Chicken Biriyani', 'Mutton Biriyani', 'Egg Biriyani', 'Vegetable Biriyani', 'Seafood Biriyani',
            'Chicken Devilled', 'Fish Devilled', 'Pork Devilled', 'Prawn Devilled', 'Squid Devilled',
            'Chicken Chopsuey', 'Mixed Chopsuey', 'Vegetable Chopsuey', 'Seafood Chopsuey', 'Egg Chopsuey',
            'Chicken Pasta', 'Seafood Pasta', 'Vegetable Pasta', 'Cheese Pasta', 'Mixed Pasta',
            'Chicken Sandwich', 'Egg Sandwich', 'Fish Sandwich', 'Cheese Sandwich', 'Vegetable Sandwich',
            'Chicken Wrap', 'Beef Wrap', 'Fish Wrap', 'Vegetable Wrap', 'Cheese Wrap',
            'Chicken Salad', 'Tuna Salad', 'Greek Salad', 'Caesar Salad', 'Garden Salad',
            'Chicken Soup', 'Vegetable Soup', 'Seafood Soup', 'Mushroom Soup', 'Tom Yum Soup',
            'Chicken Wings', 'BBQ Wings', 'Hot Wings', 'Garlic Wings', 'Honey Wings',
            'French Fries', 'Cheese Fries', 'Loaded Fries', 'Wedges', 'Onion Rings',
            'Chocolate Cake', 'Vanilla Cake', 'Red Velvet Cake', 'Cheesecake', 'Black Forest Cake',
            'Vanilla Ice Cream', 'Chocolate Ice Cream', 'Strawberry Ice Cream', 'Mango Ice Cream', 'Mixed Ice Cream',
        ];

        foreach ($productNames as $index => $name) {
            $product = Product::create([
                'category_id' => $categories->random()->id,
                'name' => $name,
                'slug' => Str::slug($name),
                'sku' => 'PRD' . str_pad($index + 1, 4, '0', STR_PAD_LEFT),
                'description' => "Delicious $name prepared with fresh ingredients",
                'image' => 'https://via.placeholder.com/400x300?text=' . urlencode($name),
                'is_active' => true,
            ]);
            $products[] = $product;
        }

        // Attach products to restaurants with random prices and settings
        foreach ($restaurants as $restaurant) {
            // Each restaurant gets 30-50 random products
            $randomProducts = collect($products)->random(rand(30, 50));
            
            foreach ($randomProducts as $product) {
                $basePrice = rand(200, 2000);
                $restaurant->products()->attach($product->id, [
                    'price' => $basePrice,
                    'offer_price' => rand(0, 1) ? $basePrice - rand(50, 200) : null,
                    'is_available' => rand(0, 10) > 1, // 90% available
                    'is_spicy' => rand(0, 1),
                    'is_vegetarian' => rand(0, 1),
                    'is_featured' => rand(0, 5) == 0, // 20% featured
                    'preparation_time' => rand(10, 45),
                    'stock_quantity' => rand(0, 1) ? rand(10, 100) : null,
                ]);
            }
        }

        $this->command->info('Created 100 products and 10 restaurants with relationships!');
    }
}
