<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Order;
use App\Models\OrderAuditTrail;
use App\Models\User;

class OrderAuditTrailSeeder extends Seeder
{
    public function run()
    {
        $this->command->info('Creating order audit trails...');
        
        $orders = Order::limit(50)->get();
        $admins = User::role(['super-admin', 'admin'])->get();
        $actionTypes = ['status_change', 'manual_override', 'refund', 'cancellation', 'assignment'];
        
        foreach ($orders as $order) {
            $actionsCount = rand(2, 5);
            
            for ($i = 0; $i < $actionsCount; $i++) {
                $admin = $admins->random();
                $actionType = $actionTypes[array_rand($actionTypes)];
                
                $auditData = match($actionType) {
                    'status_change' => [
                        'action_description' => 'Status changed from pending to accepted',
                        'old_values' => ['status' => 'pending'],
                        'new_values' => ['status' => 'accepted'],
                        'reason' => 'Restaurant confirmed order'
                    ],
                    'manual_override' => [
                        'action_description' => 'Manual status override',
                        'old_values' => ['status' => 'preparing'],
                        'new_values' => ['status' => 'ready'],
                        'reason' => 'Kitchen completed order early'
                    ],
                    'refund' => [
                        'action_description' => 'Refund processed: Rs. 1500.00',
                        'old_values' => ['refund_amount' => 0],
                        'new_values' => ['refund_amount' => 1500.00],
                        'reason' => 'Customer complaint - food quality issue'
                    ],
                    'cancellation' => [
                        'action_description' => 'Order cancelled by admin',
                        'old_values' => ['status' => 'accepted'],
                        'new_values' => ['status' => 'cancelled'],
                        'reason' => 'Restaurant temporarily closed'
                    ],
                    'assignment' => [
                        'action_description' => 'Delivery boy assigned/reassigned',
                        'old_values' => ['delivery_boy_id' => null],
                        'new_values' => ['delivery_boy_id' => $admins->random()->id],
                        'reason' => 'Manual delivery assignment'
                    ]
                };
                
                OrderAuditTrail::create([
                    'order_id' => $order->id,
                    'user_id' => $admin->id,
                    'user_role' => $admin->getRoleNames()->first(),
                    'action_type' => $actionType,
                    'action_description' => $auditData['action_description'],
                    'old_values' => $auditData['old_values'],
                    'new_values' => $auditData['new_values'],
                    'reason' => $auditData['reason'],
                    'ip_address' => '192.168.1.' . rand(1, 255),
                    'user_agent' => 'Mozilla/5.0 (Admin Dashboard)',
                    'created_at' => $order->created_at->addMinutes($i * 20),
                    'updated_at' => $order->created_at->addMinutes($i * 20)
                ]);
            }
        }
        
        $this->command->info('Order audit trails created successfully!');
    }
}