<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Restaurant;
use App\Models\User;
use App\Models\OrderAdminAction;
use App\Exceptions\ImmutableFinancialDataException;
use Illuminate\Support\Facades\DB;

class SuperAdminOrderControlService
{
    public static function forceCancel(Order $order, string $reason): array
    {
        if ($order->status === 'cancelled') {
            return ['success' => false, 'message' => 'Order is already cancelled'];
        }

        if ($order->status === 'delivered') {
            return ['success' => false, 'message' => 'Cannot cancel delivered order'];
        }

        DB::transaction(function() use ($order, $reason) {
            $order->update([
                'status' => 'cancelled',
                'cancellation_reason' => $reason,
                'cancelled_by' => 'super_admin',
                'cancelled_at' => now()
            ]);

            OrderAdminAction::create([
                'order_id' => $order->id,
                'admin_id' => request()->user()->id ?? null,
                'action' => 'force_cancel',
                'notes' => $reason
            ]);

            OrderAuditService::logCancellation($order, $reason);
        });

        return ['success' => true, 'message' => 'Order cancelled successfully'];
    }

    public static function reassignDeliveryBoy(Order $order, int $deliveryBoyId, string $reason): array
    {
        $deliveryBoy = User::role('delivery-boy')->find($deliveryBoyId);
        
        if (!$deliveryBoy) {
            return ['success' => false, 'message' => 'Invalid delivery boy selected'];
        }

        if (!in_array($order->status, ['accepted', 'preparing', 'ready_for_pickup', 'picked_up'])) {
            return ['success' => false, 'message' => 'Cannot reassign delivery for this order status'];
        }

        DB::transaction(function() use ($order, $deliveryBoyId, $deliveryBoy, $reason) {
            $oldDeliveryBoyId = $order->delivery?->delivery_boy_id;

            $order->delivery()->updateOrCreate(
                ['order_id' => $order->id],
                ['delivery_boy_id' => $deliveryBoyId, 'status' => 'assigned']
            );

            OrderAdminAction::create([
                'order_id' => $order->id,
                'admin_id' => request()->user()->id ?? null,
                'action' => 'reassign',
                'notes' => "Reassigned from delivery boy ID {$oldDeliveryBoyId} to {$deliveryBoy->name}. Reason: {$reason}"
            ]);

            OrderAuditService::logAssignment($order, $deliveryBoyId, $reason);
        });

        return ['success' => true, 'message' => 'Delivery boy reassigned successfully'];
    }

    public static function processManualRefund(Order $order, float $refundAmount, string $reason): array
    {
        if ($refundAmount <= 0 || $refundAmount > $order->total) {
            return ['success' => false, 'message' => 'Invalid refund amount'];
        }

        try {
            DB::transaction(function() use ($order, $refundAmount, $reason) {
                $order->update([
                    'refund_amount' => $refundAmount,
                    'refund_reason' => $reason,
                    'refund_status' => 'processed',
                    'refunded_by' => 'super_admin',
                    'refunded_at' => now()
                ]);

                OrderAdminAction::create([
                    'order_id' => $order->id,
                    'admin_id' => request()->user()->id ?? null,
                    'action' => 'refund',
                    'notes' => "Manual refund: Rs.{$refundAmount} - {$reason}"
                ]);

                OrderAuditService::logRefund($order, $refundAmount, $reason);

                // Update financial snapshot if not immutable
                try {
                    FinancialSnapshotService::processRefund($order, $refundAmount);
                } catch (ImmutableFinancialDataException $e) {
                    // Log but don't fail - immutable snapshots can't be updated
                    OrderAuditService::logManualOverride($order, 
                        ['refund_attempted_on_immutable' => true], 
                        'Refund processed but financial snapshot is immutable'
                    );
                }
            });

            return ['success' => true, 'message' => 'Refund processed successfully'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Refund processing failed: ' . $e->getMessage()];
        }
    }

    public static function blockRestaurant(Restaurant $restaurant, string $reason, ?int $durationHours = null): array
    {
        DB::transaction(function() use ($restaurant, $reason, $durationHours) {
            $blockedUntil = $durationHours ? now()->addHours($durationHours) : null;

            $restaurant->update([
                'is_blocked' => true,
                'blocked_reason' => $reason,
                'blocked_by' => request()->user()->id ?? null,
                'blocked_at' => now(),
                'blocked_until' => $blockedUntil
            ]);

            // Cancel all pending orders from this restaurant
            $pendingOrders = $restaurant->orders()
                ->whereIn('status', ['pending', 'accepted'])
                ->get();

            foreach ($pendingOrders as $order) {
                $order->update([
                    'status' => 'cancelled',
                    'cancellation_reason' => 'Restaurant temporarily blocked: ' . $reason,
                    'cancelled_by' => 'super_admin',
                    'cancelled_at' => now()
                ]);

                OrderAdminAction::create([
                    'order_id' => $order->id,
                    'admin_id' => request()->user()->id ?? null,
                    'action' => 'force_cancel',
                    'notes' => 'Auto-cancelled due to restaurant block: ' . $reason
                ]);
            }

            // Log the restaurant block action
            OrderAuditService::logManualOverride(
                $pendingOrders->first() ?? new Order(['id' => 0]), 
                ['restaurant_blocked' => true, 'blocked_until' => $blockedUntil], 
                "Restaurant {$restaurant->name} blocked: {$reason}"
            );
        });

        $message = $durationHours ? 
            "Restaurant blocked for {$durationHours} hours" : 
            "Restaurant blocked indefinitely";

        return ['success' => true, 'message' => $message];
    }

    public static function unblockRestaurant(Restaurant $restaurant, string $reason): array
    {
        if (!$restaurant->is_blocked) {
            return ['success' => false, 'message' => 'Restaurant is not blocked'];
        }

        $restaurant->update([
            'is_blocked' => false,
            'blocked_reason' => null,
            'blocked_by' => null,
            'blocked_at' => null,
            'blocked_until' => null,
            'unblocked_by' => request()->user()->id ?? null,
            'unblocked_at' => now(),
            'unblock_reason' => $reason
        ]);

        OrderAuditService::logManualOverride(
            new Order(['id' => 0]), 
            ['restaurant_unblocked' => true], 
            "Restaurant {$restaurant->name} unblocked: {$reason}"
        );

        return ['success' => true, 'message' => 'Restaurant unblocked successfully'];
    }

    public static function getAvailableDeliveryBoys(Order $order): array
    {
        return User::role('delivery-boy')
            ->where('is_active', true)
            ->whereDoesntHave('deliveries', function($q) {
                $q->whereIn('status', ['assigned', 'picked_up']);
            })
            ->select('id', 'name', 'phone')
            ->get()
            ->toArray();
    }

    public static function getBlockedRestaurants(): array
    {
        return Restaurant::where('is_blocked', true)
            ->with(['blockedBy:id,name'])
            ->select('id', 'name', 'blocked_reason', 'blocked_by', 'blocked_at', 'blocked_until')
            ->get()
            ->toArray();
    }
}