<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderAuditTrail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class OrderAuditService
{
    public static function logStatusChange(Order $order, string $oldStatus, string $newStatus, ?string $reason = null): void
    {
        self::createAuditLog($order, [
            'action_type' => 'status_change',
            'action_description' => "Status changed from '{$oldStatus}' to '{$newStatus}'",
            'old_values' => ['status' => $oldStatus],
            'new_values' => ['status' => $newStatus],
            'reason' => $reason
        ]);
    }

    public static function logManualOverride(Order $order, array $changes, string $reason): void
    {
        $description = 'Manual override: ' . implode(', ', array_keys($changes));
        
        self::createAuditLog($order, [
            'action_type' => 'manual_override',
            'action_description' => $description,
            'old_values' => $order->getOriginal(),
            'new_values' => $changes,
            'reason' => $reason
        ]);
    }

    public static function logRefund(Order $order, float $amount, string $reason): void
    {
        self::createAuditLog($order, [
            'action_type' => 'refund',
            'action_description' => "Refund processed: Rs. {$amount}",
            'old_values' => ['refund_amount' => $order->getOriginal('refund_amount') ?? 0],
            'new_values' => ['refund_amount' => $amount],
            'reason' => $reason
        ]);
    }

    public static function logCancellation(Order $order, string $reason): void
    {
        self::createAuditLog($order, [
            'action_type' => 'cancellation',
            'action_description' => 'Order cancelled by admin',
            'old_values' => ['status' => $order->getOriginal('status')],
            'new_values' => ['status' => 'cancelled'],
            'reason' => $reason
        ]);
    }

    public static function logAssignment(Order $order, int $deliveryBoyId, ?string $reason = null): void
    {
        self::createAuditLog($order, [
            'action_type' => 'assignment',
            'action_description' => "Delivery boy assigned/reassigned",
            'old_values' => ['delivery_boy_id' => $order->delivery?->delivery_boy_id],
            'new_values' => ['delivery_boy_id' => $deliveryBoyId],
            'reason' => $reason ?? 'Delivery assignment'
        ]);
    }

    public static function logFlagAction(Order $order, string $flagType, string $action, ?string $reason = null): void
    {
        self::createAuditLog($order, [
            'action_type' => 'flag_action',
            'action_description' => "Flag {$action}: {$flagType}",
            'old_values' => null,
            'new_values' => ['flag_type' => $flagType, 'flag_action' => $action],
            'reason' => $reason
        ]);
    }

    private static function createAuditLog(Order $order, array $data): void
    {
        $user = request()->user();
        $request = request();

        OrderAuditTrail::create(array_merge($data, [
            'order_id' => $order->id,
            'user_id' => $user?->id ?? 1,
            'user_role' => $user?->roles->pluck('name')->first() ?? 'system',
            'ip_address' => $request?->ip(),
            'user_agent' => $request?->userAgent()
        ]));
    }

    public static function getOrderAuditTrail(Order $order): \Illuminate\Database\Eloquent\Collection
    {
        return $order->auditTrails()
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public static function getAuditSummary(int $days = 30): array
    {
        $trails = OrderAuditTrail::recent($days)->get();

        return [
            'total_actions' => $trails->count(),
            'by_action_type' => $trails->groupBy('action_type')->map->count(),
            'by_user' => $trails->groupBy('user_id')->map->count(),
            'recent_activity' => $trails->take(10)->map(function ($trail) {
                return [
                    'order_number' => $trail->order->order_number,
                    'action' => $trail->action_description,
                    'user' => $trail->user->name,
                    'timestamp' => $trail->created_at
                ];
            })
        ];
    }
}