<?php

namespace App\Services;

use App\Models\DeliveryBoy;
use App\Models\DeliveryBoyLocation;
use App\Models\Delivery;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class LocationTrackingService
{
    /**
     * Update delivery boy location
     */
    public function updateLocation(
        DeliveryBoy $deliveryBoy,
        float $latitude,
        float $longitude,
        ?array $metadata = []
    ): DeliveryBoyLocation {
        // Update delivery boy current location
        $deliveryBoy->update([
            'latitude' => $latitude,
            'longitude' => $longitude,
            'last_active_at' => now()
        ]);

        // Get active delivery
        $activeDelivery = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->whereIn('status', ['assigned', 'accepted', 'picked_up'])
            ->first();

        // Record location history
        $location = DeliveryBoyLocation::create([
            'delivery_boy_id' => $deliveryBoy->id,
            'delivery_id' => $activeDelivery?->id,
            'latitude' => $latitude,
            'longitude' => $longitude,
            'accuracy' => $metadata['accuracy'] ?? null,
            'speed' => $metadata['speed'] ?? null,
            'heading' => $metadata['heading'] ?? null,
            'address' => $metadata['address'] ?? null,
            'recorded_at' => now()
        ]);

        // Cache current location for quick access (5 minutes)
        Cache::put(
            "delivery_boy_location:{$deliveryBoy->id}",
            [
                'latitude' => $latitude,
                'longitude' => $longitude,
                'updated_at' => now()->toIso8601String()
            ],
            300
        );

        return $location;
    }

    /**
     * Get current location from cache or database
     */
    public function getCurrentLocation(DeliveryBoy $deliveryBoy): ?array
    {
        return Cache::remember(
            "delivery_boy_location:{$deliveryBoy->id}",
            300,
            fn() => [
                'latitude' => $deliveryBoy->latitude,
                'longitude' => $deliveryBoy->longitude,
                'updated_at' => $deliveryBoy->last_active_at?->toIso8601String()
            ]
        );
    }

    /**
     * Get location history for delivery
     */
    public function getDeliveryRoute(Delivery $delivery): array
    {
        $locations = DeliveryBoyLocation::where('delivery_id', $delivery->id)
            ->orderBy('recorded_at')
            ->get(['latitude', 'longitude', 'recorded_at']);

        return [
            'delivery_id' => $delivery->id,
            'route' => $locations,
            'total_points' => $locations->count(),
            'start_time' => $locations->first()?->recorded_at,
            'last_update' => $locations->last()?->recorded_at,
        ];
    }

    /**
     * Calculate distance between two points (Haversine formula)
     */
    public function calculateDistance(
        float $lat1,
        float $lon1,
        float $lat2,
        float $lon2
    ): float {
        $earthRadius = 6371; // km

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat / 2) * sin($dLat / 2) +
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
             sin($dLon / 2) * sin($dLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        return $earthRadius * $c;
    }

    /**
     * Get nearby delivery boys
     */
    public function getNearbyDeliveryBoys(
        float $latitude,
        float $longitude,
        float $radiusKm = 5
    ): array {
        $deliveryBoys = DeliveryBoy::where('status', 'online')
            ->where('is_available', true)
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->get();

        return $deliveryBoys->filter(function ($deliveryBoy) use ($latitude, $longitude, $radiusKm) {
            $distance = $this->calculateDistance(
                $latitude,
                $longitude,
                $deliveryBoy->latitude,
                $deliveryBoy->longitude
            );
            return $distance <= $radiusKm;
        })->map(function ($deliveryBoy) use ($latitude, $longitude) {
            return [
                'id' => $deliveryBoy->id,
                'name' => $deliveryBoy->full_name,
                'latitude' => $deliveryBoy->latitude,
                'longitude' => $deliveryBoy->longitude,
                'distance' => $this->calculateDistance(
                    $latitude,
                    $longitude,
                    $deliveryBoy->latitude,
                    $deliveryBoy->longitude
                ),
                'rating' => $deliveryBoy->rating,
            ];
        })->sortBy('distance')->values()->all();
    }

    /**
     * Clean old location records (keep last 30 days)
     */
    public function cleanOldLocations(): int
    {
        return DeliveryBoyLocation::where('recorded_at', '<', Carbon::now()->subDays(30))
            ->delete();
    }
}
