<?php

namespace App\Services;

use App\Models\ActivityLog;
use Illuminate\Database\Eloquent\Model;

class ActivityLoggerService
{
    public function log(
        string $logType,
        string $action,
        Model $subject,
        ?array $properties = null
    ): ActivityLog {
        return ActivityLog::create([
            'log_type' => $logType,
            'action' => $action,
            'user_id' => request()->user()->id ?? null,
            'subject_type' => get_class($subject),
            'subject_id' => $subject->id,
            'properties' => $properties,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }

    // Delivery Boy Events
    public function deliveryBoyCreated($deliveryBoy): ActivityLog
    {
        return $this->log('delivery_boy', 'created', $deliveryBoy, [
            'name' => $deliveryBoy->full_name,
            'phone' => $deliveryBoy->phone,
            'email' => $deliveryBoy->email,
        ]);
    }

    public function deliveryBoyUpdated($deliveryBoy, array $changes): ActivityLog
    {
        return $this->log('delivery_boy', 'updated', $deliveryBoy, [
            'changes' => $changes,
        ]);
    }

    public function deliveryBoyBlocked($deliveryBoy, string $reason = null): ActivityLog
    {
        return $this->log('delivery_boy', 'blocked', $deliveryBoy, [
            'reason' => $reason,
        ]);
    }

    public function deliveryBoyUnblocked($deliveryBoy): ActivityLog
    {
        return $this->log('delivery_boy', 'unblocked', $deliveryBoy);
    }

    public function deliveryBoyVerified($deliveryBoy): ActivityLog
    {
        return $this->log('delivery_boy', 'verified', $deliveryBoy);
    }

    public function deliveryBoyRejected($deliveryBoy, string $reason): ActivityLog
    {
        return $this->log('delivery_boy', 'rejected', $deliveryBoy, [
            'reason' => $reason,
        ]);
    }

    public function deliveryBoyDeleted($deliveryBoy): ActivityLog
    {
        return $this->log('delivery_boy', 'deleted', $deliveryBoy, [
            'name' => $deliveryBoy->full_name,
        ]);
    }

    // Earning Events
    public function earningRecorded($earning): ActivityLog
    {
        return $this->log('earning', 'recorded', $earning, [
            'delivery_boy_id' => $earning->delivery_boy_id,
            'amount' => $earning->amount,
            'type' => $earning->earning_type,
        ]);
    }

    public function payoutProcessed($deliveryBoy, float $amount, string $method): ActivityLog
    {
        return $this->log('payout', 'processed', $deliveryBoy, [
            'amount' => $amount,
            'method' => $method,
        ]);
    }

    // Location Events
    public function locationUpdated($deliveryBoy, array $location): ActivityLog
    {
        return $this->log('location', 'updated', $deliveryBoy, [
            'latitude' => $location['latitude'],
            'longitude' => $location['longitude'],
        ]);
    }

    // Area Events
    public function areaAssigned($deliveryBoy, $area): ActivityLog
    {
        return $this->log('area', 'assigned', $deliveryBoy, [
            'area_name' => $area->area_name,
            'city' => $area->city,
        ]);
    }
}
