<?php

namespace App\Repositories;

use App\Models\Restaurant;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class RestaurantRepository
{
    public function create(array $data)
    {
        return Restaurant::create($data);
    }

    public function update(Restaurant $restaurant, array $data)
    {
        $restaurant->update($data);
        return $restaurant->fresh();
    }

    public function delete(Restaurant $restaurant)
    {
        return $restaurant->delete();
    }

    public function findById($id)
    {
        return Restaurant::with([
            'user',
            'owner',
            'restaurantCategories',
            'commissionRule',
            'orders',
            'reviews'
        ])->findOrFail($id);
    }

    public function search(array $filters = [])
    {
        $query = Restaurant::with(['user', 'owner', 'restaurantCategories', 'commissionRule']);

        // Apply filters
        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['is_featured'])) {
            $query->where('is_featured', $filters['is_featured']);
        }

        if (isset($filters['category_id'])) {
            $query->whereHas('restaurantCategories', function (Builder $q) use ($filters) {
                $q->where('restaurant_categories.id', $filters['category_id']);
            });
        }

        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function (Builder $q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        if (isset($filters['owner_id'])) {
            $query->where('owner_id', $filters['owner_id']);
        }

        // Apply sorting
        $sortBy = $filters['sort_by'] ?? 'created_at';
        $sortOrder = $filters['sort_order'] ?? 'desc';
        $query->orderBy($sortBy, $sortOrder);

        // Apply pagination
        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    public function getActiveRestaurants()
    {
        return Restaurant::where('status', 'active')
            ->with(['restaurantCategories', 'commissionRule'])
            ->orderBy('name')
            ->get();
    }

    public function getFeaturedRestaurants()
    {
        return Restaurant::where('status', 'active')
            ->where('is_featured', true)
            ->with(['restaurantCategories', 'commissionRule'])
            ->orderBy('name')
            ->get();
    }

    public function getPendingRestaurants()
    {
        return Restaurant::where('status', 'pending')
            ->with(['user', 'restaurantCategories'])
            ->orderBy('created_at', 'desc')
            ->get();
    }



    public function getTopRestaurants(int $limit = 10)
    {
        return Restaurant::select('restaurants.*')
            ->selectRaw('(
                SELECT COUNT(*) 
                FROM orders 
                WHERE orders.restaurant_id = restaurants.id 
                AND orders.status = "delivered"
            ) as total_orders')
            ->selectRaw('(
                SELECT SUM(total_amount) 
                FROM orders 
                WHERE orders.restaurant_id = restaurants.id 
                AND orders.status = "delivered"
            ) as total_revenue')
            ->where('status', 'active')
            ->orderByDesc('total_revenue')
            ->limit($limit)
            ->get();
    }

    public function getRestaurantsByCategory(int $categoryId)
    {
        return Restaurant::whereHas('restaurantCategories', function (Builder $query) use ($categoryId) {
            $query->where('restaurant_categories.id', $categoryId);
        })
        ->where('status', 'active')
        ->with(['restaurantCategories', 'commissionRule'])
        ->orderBy('name')
        ->get();
    }

    public function getRestaurantsNearLocation(float $latitude, float $longitude, float $radius = 10)
    {
        return Restaurant::select('restaurants.*')
            ->selectRaw('(
                6371 * acos(
                    cos(radians(?)) * cos(radians(latitude)) * 
                    cos(radians(longitude) - radians(?)) + 
                    sin(radians(?)) * sin(radians(latitude))
                )
            ) AS distance', [$latitude, $longitude, $latitude])
            ->having('distance', '<', $radius)
            ->where('status', 'active')
            ->orderBy('distance')
            ->with(['restaurantCategories', 'commissionRule'])
            ->get();
    }

    public function getPendingApproval(int $perPage = 15)
    {
        return Restaurant::with(['user'])
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
    }

    public function getRestaurantStats(Restaurant $restaurant, Carbon $startDate = null, Carbon $endDate = null)
    {
        $startDate = $startDate ?? now()->subDays(30);
        $endDate = $endDate ?? now();

        return [
            'total_orders' => $restaurant->orders()->whereBetween('created_at', [$startDate, $endDate])->count(),
            'total_revenue' => $restaurant->orders()->whereBetween('created_at', [$startDate, $endDate])->sum('total'),
            'average_order_value' => $restaurant->orders()->whereBetween('created_at', [$startDate, $endDate])->avg('total') ?? 0,
            'total_products' => $restaurant->products()->count(),
            'rating' => $restaurant->rating,
            'total_reviews' => $restaurant->total_reviews,
        ];
    }
}