<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\OrderRiskDetectionService;
use Illuminate\Support\Facades\Log;

class ProcessRiskDetection implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private string $detectionType;

    public function __construct(string $detectionType = 'all')
    {
        $this->detectionType = $detectionType;
    }

    public function handle(): void
    {
        try {
            $results = match($this->detectionType) {
                'late' => ['late_deliveries' => OrderRiskDetectionService::detectLateDeliveries()],
                'cancellation' => ['high_cancellation_restaurants' => OrderRiskDetectionService::detectHighCancellationRestaurants()],
                'rejection' => ['delivery_boy_rejections' => OrderRiskDetectionService::detectDeliveryBoyRejections()],
                'payment' => ['payment_mismatches' => OrderRiskDetectionService::detectPaymentMismatches()],
                default => OrderRiskDetectionService::runAllDetections()
            };

            $totalIssues = collect($results)->except('detection_timestamp')->sum(function($data) {
                return is_array($data) ? count($data) : 0;
            });

            Log::info("Risk detection completed", [
                'type' => $this->detectionType,
                'issues_detected' => $totalIssues,
                'timestamp' => now()
            ]);

        } catch (\Exception $e) {
            Log::error("Risk detection failed", [
                'type' => $this->detectionType,
                'error' => $e->getMessage(),
                'timestamp' => now()
            ]);
            
            throw $e;
        }
    }
}