<?php

namespace App\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateDeliveryBoyRequest extends FormRequest
{
    public function authorize(): bool
    {
        return $this->user()->hasAnyRole(['super-admin', 'delivery-boy']);
    }

    public function rules(): array
    {
        $deliveryBoyId = $this->route('deliveryBoy')->id;

        return [
            'full_name' => 'sometimes|string|max:255',
            'phone' => [
                'sometimes',
                'string',
                'max:20',
                Rule::unique('delivery_boys', 'phone')->ignore($deliveryBoyId)
            ],
            'email' => [
                'sometimes',
                'email',
                'max:255',
                Rule::unique('delivery_boys', 'email')->ignore($deliveryBoyId)
            ],
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'national_id' => 'nullable|string|max:50',
            'date_of_birth' => 'nullable|date|before:today',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'commission_rate' => 'nullable|numeric|min:0|max:100',
            'is_available' => 'nullable|boolean',
            'status' => 'sometimes|in:online,offline,on_delivery,blocked',
        ];
    }

    public function messages(): array
    {
        return [
            'phone.unique' => 'Phone number already exists',
            'email.unique' => 'Email already exists',
            'date_of_birth.before' => 'Date of birth must be before today',
            'commission_rate.max' => 'Commission rate cannot exceed 100%',
            'status.in' => 'Invalid status value',
        ];
    }
}
