<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Restaurant;
use Illuminate\Http\Request;

class RestaurantManagementController extends Controller
{
    public function index()
    {
        $restaurants = Restaurant::with(['owner', 'categories'])
            ->latest()
            ->paginate(20);

        return response()->json($restaurants);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'address' => 'required|string',
            'phone' => 'required|string',
            'email' => 'required|email',
            'cuisine_type' => 'required|string',
            'opening_time' => 'required|date_format:H:i',
            'closing_time' => 'required|date_format:H:i',
            'delivery_radius' => 'required|numeric|min:0',
            'delivery_fee' => 'required|numeric|min:0',
            'minimum_order' => 'required|numeric|min:0',
            'commission_rate' => 'required|numeric|min:0|max:100',
        ]);

        $restaurant = Restaurant::create($validated + [
            'owner_id' => $request->user()->id,
            'status' => 'pending',
        ]);

        return response()->json($restaurant, 201);
    }

    public function show(Restaurant $restaurant)
    {
        return response()->json($restaurant->load(['owner', 'categories', 'products']));
    }

    public function update(Request $request, Restaurant $restaurant)
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'address' => 'sometimes|string',
            'phone' => 'sometimes|string',
            'email' => 'sometimes|email',
            'cuisine_type' => 'sometimes|string',
            'opening_time' => 'sometimes|date_format:H:i',
            'closing_time' => 'sometimes|date_format:H:i',
            'delivery_radius' => 'sometimes|numeric|min:0',
            'delivery_fee' => 'sometimes|numeric|min:0',
            'minimum_order' => 'sometimes|numeric|min:0',
            'commission_rate' => 'sometimes|numeric|min:0|max:100',
        ]);

        $restaurant->update($validated);

        return response()->json($restaurant);
    }

    public function destroy(Restaurant $restaurant)
    {
        $restaurant->delete();
        return response()->json(null, 204);
    }

    public function approve(Restaurant $restaurant)
    {
        $restaurant->update(['status' => 'approved']);
        return response()->json($restaurant);
    }

    public function suspend(Restaurant $restaurant)
    {
        $restaurant->update(['status' => 'suspended']);
        return response()->json($restaurant);
    }

    public function restore(Restaurant $restaurant)
    {
        $restaurant->update(['status' => 'approved']);
        return response()->json($restaurant);
    }

    public function analytics(Restaurant $restaurant)
    {
        return response()->json([
            'total_orders' => $restaurant->orders()->count(),
            'total_revenue' => $restaurant->orders()->sum('total'),
            'average_order_value' => $restaurant->orders()->avg('total'),
        ]);
    }

    public function orders(Restaurant $restaurant)
    {
        $orders = $restaurant->orders()
            ->with(['user', 'products'])
            ->latest()
            ->paginate(20);

        return response()->json($orders);
    }

    public function products(Restaurant $restaurant)
    {
        $products = $restaurant->products()
            ->latest()
            ->paginate(20);

        return response()->json($products);
    }

    public function updateCommission(Request $request, Restaurant $restaurant)
    {
        $validated = $request->validate([
            'commission_rate' => 'required|numeric|min:0|max:100',
        ]);

        $restaurant->update(['commission_rate' => $validated['commission_rate']]);

        return response()->json($restaurant);
    }
}