<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Cart;
use App\Models\Restaurant;
use App\Services\MapService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::with(['user', 'restaurant', 'items.product', 'delivery.deliveryBoy', 'reviews']);

        $userRoles = $request->user()->roles->pluck('name')->toArray();
        if (!in_array('admin', $userRoles) && !in_array('super-admin', $userRoles)) {
            $query->where('user_id', $request->user()->id);
        }

        $orders = $query->latest()->paginate(15);
        
        // Add rating to each order from reviews
        $orders->getCollection()->transform(function ($order) {
            $review = $order->reviews->where('reviewable_type', 'App\\Models\\Restaurant')->first();
            $order->rating = $review ? $review->rating : null;
            $order->rating_comment = $review ? $review->comment : null;
            return $order;
        });

        return response()->json($orders);
    }

    public function store(Request $request)
    {
        $request->validate([
            'restaurant_id' => 'required|exists:restaurants,id',
            'delivery_address' => 'required|string',
            'delivery_latitude' => 'required|numeric',
            'delivery_longitude' => 'required|numeric',
            'delivery_phone' => 'required|string',
            'payment_method' => 'required|in:cash,card,wallet',
            'delivery_fee' => 'nullable|numeric',
        ]);

        $carts = Cart::where('user_id', $request->user()->id)
            ->where('restaurant_id', $request->restaurant_id)
            ->get();

        if ($carts->isEmpty()) {
            return response()->json(['message' => 'Cart is empty'], 400);
        }

        $restaurant = Restaurant::findOrFail($request->restaurant_id);
        
        // Calculate distance using Haversine formula (same as frontend)
        $distance = $this->calculateDistance(
            $restaurant->latitude,
            $restaurant->longitude,
            $request->delivery_latitude,
            $request->delivery_longitude
        );
        
        $estimatedTime = ceil($distance * 3); // 3 minutes per km
        
        $subtotal = $carts->sum('total_price');
        // Use frontend calculated delivery fee if provided, otherwise calculate
        $deliveryFee = $request->delivery_fee ?? ceil($restaurant->delivery_fee * $distance);
        $serviceCharge = $subtotal * ($restaurant->service_charge_percentage / 100);
        $tax = $subtotal * ($restaurant->tax_percentage / 100);
        $total = $subtotal + $deliveryFee + $serviceCharge + $tax;

        $order = Order::create([
            'order_number' => 'ORD-' . strtoupper(Str::random(8)),
            'user_id' => $request->user()->id,
            'restaurant_id' => $request->restaurant_id,
            'delivery_address' => $request->delivery_address,
            'delivery_latitude' => $request->delivery_latitude,
            'delivery_longitude' => $request->delivery_longitude,
            'delivery_phone' => $request->delivery_phone,
            'delivery_instructions' => $request->delivery_instructions,
            'subtotal' => $subtotal,
            'delivery_fee' => $deliveryFee,
            'service_charge' => $serviceCharge,
            'tax' => $tax,
            'total' => $total,
            'distance_km' => $distance,
            'estimated_delivery_time' => $estimatedTime,
            'payment_method' => $request->payment_method,
            'customer_notes' => $request->customer_notes,
        ]);

        foreach ($carts as $cart) {
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $cart->product_id,
                'product_name' => $cart->product->name,
                'quantity' => $cart->quantity,
                'unit_price' => $cart->unit_price,
                'variations' => $cart->variations,
                'addons' => $cart->addons,
                'special_notes' => $cart->special_notes,
                'total_price' => $cart->total_price,
            ]);
        }

        Cart::where('user_id', $request->user()->id)
            ->where('restaurant_id', $request->restaurant_id)
            ->delete();

        return response()->json($order->load('items'), 201);
    }

    public function show($id)
    {
        $order = Order::with(['user', 'restaurant', 'items.product', 'delivery.deliveryBoy', 'payment'])
            ->findOrFail($id);

        return response()->json($order);
    }

    public function updateStatus(Request $request, $id)
    {
        try {
            $request->validate([
                'status' => 'required|in:accepted,preparing,cooking,ready,picked_up,on_the_way,delivered,completed,cancelled',
            ]);

            $order = Order::findOrFail($id);
            
            // Prevent changing status of delivered or cancelled orders
            if (in_array($order->status, ['delivered', 'cancelled'])) {
                return response()->json([
                    'message' => 'Cannot change status of ' . $order->status . ' orders'
                ], 400);
            }
            
            // Store old status for event
            $oldStatus = $order->status;
            
            // Check if order is already in this status
            $nextStatus = $request->status === 'cooking' ? 'preparing' : $request->status;
            $alreadyInStatus = ($order->status === $nextStatus);
            
            if ($alreadyInStatus && $nextStatus !== 'ready') {
                return response()->json($order); // Return success if already in this status (except ready)
            }
            
            // Check if manager can update this order
            $user = $request->user();
            $userRoles = $user->roles->pluck('name')->toArray();
            
            if (in_array('manager', $userRoles)) {
                if (!$user->restaurant_id) {
                    return response()->json(['message' => 'No restaurant assigned to this manager'], 404);
                }
                if ($order->restaurant_id !== $user->restaurant_id) {
                    return response()->json(['message' => 'Unauthorized - This order does not belong to your restaurant'], 403);
                }
            }

            if (!$alreadyInStatus) {
                $order->update(['status' => $nextStatus]);
            }

            if ($nextStatus === 'accepted' && !$alreadyInStatus) {
                $order->update(['accepted_at' => now()]);
            } elseif ($nextStatus === 'preparing' && !$alreadyInStatus) {
                $order->update(['preparing_at' => now()]);
            } elseif ($nextStatus === 'ready') {
                if (!$alreadyInStatus) {
                    $order->update(['ready_at' => now()]);
                }
                
                // Auto-assign delivery boy when order is ready (even if already ready)
                $this->autoAssignDeliveryBoy($order);
            } elseif ($nextStatus === 'picked_up' && !$alreadyInStatus) {
                $order->update(['picked_up_at' => now()]);
            } elseif ($nextStatus === 'delivered' && !$alreadyInStatus) {
                $order->update(['delivered_at' => now()]);
                
                // Notify delivery boy about successful delivery
                if ($order->delivery && $order->delivery->delivery_boy_id) {
                    $deliveryBoy = \App\Models\User::find($order->delivery->delivery_boy_id);
                    if ($deliveryBoy) {
                        \App\Models\Notification::create([
                            'user_id' => $deliveryBoy->id,
                            'type' => 'delivery_completed',
                            'title' => 'Delivery Completed',
                            'message' => "Order #{$order->order_number} has been marked as delivered. Great job!",
                            'data' => json_encode(['order_id' => $order->id]),
                        ]);
                    }
                }
            }

            // Fire event for delivery boy assignment
            event(new \App\Events\OrderStatusChanged($order, $oldStatus));

            return response()->json($order);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error updating order status',
                'error' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => basename($e->getFile())
            ], 500);
        }
    }

    public function cancel(Request $request, $id)
    {
        $request->validate([
            'cancellation_reason' => 'required|string|min:10|max:500',
        ]);

        $order = Order::findOrFail($id);

        if (!in_array($order->status, ['pending', 'accepted'])) {
            return response()->json(['message' => 'Cannot cancel order at this stage'], 400);
        }

        $order->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancelled_by' => $request->user()->id,
            'cancellation_reason' => $request->cancellation_reason,
        ]);

        return response()->json($order);
    }

    public function invoice(Request $request, $id)
    {
        $order = Order::with(['restaurant', 'items.product', 'payment'])->findOrFail($id);
        $userRoles = $request->user()->roles->pluck('name')->toArray();
        if (!in_array('admin', $userRoles) && !in_array('super-admin', $userRoles) && $order->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }
        return response()->json([
            'order_number' => $order->order_number,
            'date' => $order->created_at,
            'customer' => $order->user_id,
            'restaurant' => $order->restaurant->name,
            'items' => $order->items->map(function ($item) {
                return [
                    'name' => $item->product_name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'total_price' => $item->total_price,
                ];
            }),
            'subtotal' => $order->subtotal,
            'delivery_fee' => $order->delivery_fee,
            'service_charge' => $order->service_charge,
            'tax' => $order->tax,
            'discount' => $order->discount,
            'total' => $order->total,
            'payment_method' => $order->payment_method,
            'payment_status' => $order->payment_status,
            'payment' => $order->payment,
        ]);
    }

    public function rate(Request $request, $id)
    {
        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:500',
        ]);

        $order = Order::findOrFail($id);

        // Check if user owns this order
        if ($order->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Check if order is delivered
        if ($order->status !== 'delivered') {
            return response()->json(['message' => 'Can only rate delivered orders'], 400);
        }

        // Check if already rated
        $existingReview = \App\Models\Review::where('order_id', $order->id)
            ->where('user_id', $request->user()->id)
            ->where('reviewable_type', 'App\\Models\\Restaurant')
            ->where('reviewable_id', $order->restaurant_id)
            ->first();

        if ($existingReview) {
            return response()->json(['message' => 'Order already rated'], 400);
        }

        // Create review
        $review = \App\Models\Review::create([
            'user_id' => $request->user()->id,
            'order_id' => $order->id,
            'reviewable_type' => 'App\\Models\\Restaurant',
            'reviewable_id' => $order->restaurant_id,
            'rating' => $request->rating,
            'comment' => $request->comment,
        ]);

        return response()->json($review);
    }

    private function autoAssignDeliveryBoy($order)
    {
        try {
            // Check if already assigned
            if ($order->delivery && $order->delivery->delivery_boy_id) {
                return;
            }

            // Get restaurant location
            $restaurant = $order->restaurant;

            // Find available delivery boys in same city
            $deliveryBoys = \App\Models\DeliveryBoy::with('user')
                ->join('delivery_boy_areas', 'delivery_boys.id', '=', 'delivery_boy_areas.delivery_boy_id')
                ->where('delivery_boys.is_available', true)
                ->where('delivery_boy_areas.city', $restaurant->city)
                ->where('delivery_boy_areas.is_active', true)
                ->select('delivery_boys.*')
                ->distinct()
                ->get();

            if ($deliveryBoys->isEmpty()) {
                return;
            }

            // Send notification to all delivery boys in same city
            foreach ($deliveryBoys as $boy) {
                // Calculate distance from restaurant to customer
                $distance = null;
                if ($order->delivery_latitude && $order->delivery_longitude && $restaurant->latitude && $restaurant->longitude) {
                    $distance = $this->calculateDistance(
                        $restaurant->latitude,
                        $restaurant->longitude,
                        $order->delivery_latitude,
                        $order->delivery_longitude
                    );
                }
                
                // Create delivery notification
                \DB::table('delivery_notifications')->insert([
                    'order_id' => $order->id,
                    'delivery_boy_id' => $boy->id,
                    'distance' => $distance,
                    'status' => 'pending',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                // Create notification for delivery boy
                \App\Models\User::find($boy->user_id)->notify(
                    new \App\Notifications\NewDeliveryAssignment($order)
                );
            }

            // Increment assignment attempts
            $order->increment('assignment_attempts');
        } catch (\Exception $e) {
            \Log::error('Auto-assign delivery boy failed: ' . $e->getMessage());
        }
    }

    public function notifyDeliveryBoys(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        
        // Check if manager can access this order
        $user = $request->user();
        $userRoles = $user->roles->pluck('name')->toArray();
        
        if (in_array('manager', $userRoles)) {
            if (!$user->restaurant_id || $order->restaurant_id !== $user->restaurant_id) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        }
        
        $this->autoAssignDeliveryBoy($order);
        
        return response()->json(['message' => 'Notifications sent successfully']);
    }

    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        $a = sin($dLat/2) * sin($dLat/2) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * sin($dLon/2) * sin($dLon/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        return $earthRadius * $c;
    }
}
