<?php

namespace App\Http\Controllers\API\Manager;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;

class RestaurantProductController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'price' => 'required|numeric|min:0',
            'offer_price' => 'nullable|numeric|min:0',
            'stock_quantity' => 'nullable|integer|min:0',
            'preparation_time' => 'nullable|integer|min:0',
            'is_vegetarian' => 'boolean',
            'is_spicy' => 'boolean',
            'is_featured' => 'boolean',
            'variations' => 'nullable|array',
            'variations.*.name' => 'required|string',
            'variations.*.price' => 'required|numeric',
            'addons' => 'nullable|array',
            'addons.*.name' => 'required|string',
            'addons.*.price' => 'required|numeric'
        ]);

        $restaurant = $request->user()->restaurants()->first();
        
        if (!$restaurant) {
            return response()->json(['message' => 'No restaurant found'], 404);
        }

        // Check if already added
        if ($restaurant->products()->where('product_id', $validated['product_id'])->exists()) {
            return response()->json(['message' => 'Product already added to your menu'], 422);
        }

        $restaurant->products()->attach($validated['product_id'], [
            'price' => $validated['price'],
            'offer_price' => $validated['offer_price'] ?? null,
            'stock_quantity' => $validated['stock_quantity'] ?? 100,
            'preparation_time' => $validated['preparation_time'] ?? 15,
            'is_vegetarian' => $validated['is_vegetarian'] ?? false,
            'is_spicy' => $validated['is_spicy'] ?? false,
            'is_featured' => $validated['is_featured'] ?? false,
            'is_available' => true
        ]);

        $product = Product::find($validated['product_id']);

        // Save variations
        if (!empty($validated['variations'])) {
            foreach ($validated['variations'] as $variation) {
                if (!empty($variation['name'])) {
                    $product->variations()->create([
                        'name' => $variation['name'],
                        'value' => $variation['name'],
                        'price' => $variation['price'] ?? 0
                    ]);
                }
            }
        }

        // Save addons
        if (!empty($validated['addons'])) {
            foreach ($validated['addons'] as $addon) {
                if (!empty($addon['name'])) {
                    $product->addons()->create([
                        'name' => $addon['name'],
                        'price' => $addon['price'] ?? 0
                    ]);
                }
            }
        }

        return response()->json(['message' => 'Product added to your menu successfully']);
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'price' => 'nullable|numeric|min:0',
            'offer_price' => 'nullable|numeric|min:0',
            'stock_quantity' => 'nullable|integer|min:0',
            'preparation_time' => 'nullable|integer|min:0',
            'is_vegetarian' => 'nullable',
            'is_spicy' => 'nullable',
            'is_featured' => 'nullable',
            'is_available' => 'nullable',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'special_notes' => 'nullable|string',
            'available_from' => 'nullable|string',
            'available_until' => 'nullable|string',
            'min_order_quantity' => 'nullable|integer|min:1',
            'variations' => 'nullable|array',
            'variations.*.name' => 'required|string',
            'variations.*.price' => 'required|numeric',
            'addons' => 'nullable|array',
            'addons.*.name' => 'required|string',
            'addons.*.price' => 'required|numeric'
        ]);

        $restaurant = $request->user()->restaurants()->first();
        
        if (!$restaurant) {
            return response()->json(['message' => 'No restaurant found'], 404);
        }

        // Prepare update data
        $updateData = [];
        foreach ($validated as $key => $value) {
            if (in_array($key, ['variations', 'addons'])) continue;
            
            // Skip null values but allow empty strings for time fields
            if ($value !== null || in_array($key, ['available_from', 'available_until', 'special_notes'])) {
                // Convert boolean fields
                if (in_array($key, ['is_vegetarian', 'is_spicy', 'is_featured', 'is_available'])) {
                    $updateData[$key] = (bool)$value;
                } else {
                    $updateData[$key] = $value;
                }
            }
        }
        
        \Log::info('Updating product pivot data:', ['product_id' => $id, 'data' => $updateData]);
        
        $restaurant->products()->updateExistingPivot($id, $updateData);

        $product = Product::find($id);

        // Update variations
        if (isset($validated['variations'])) {
            $product->variations()->delete();
            foreach ($validated['variations'] as $variation) {
                if (!empty($variation['name'])) {
                    $product->variations()->create([
                        'name' => $variation['name'],
                        'value' => $variation['name'],
                        'price' => $variation['price'] ?? 0
                    ]);
                }
            }
        }

        // Update addons
        if (isset($validated['addons'])) {
            $product->addons()->delete();
            foreach ($validated['addons'] as $addon) {
                if (!empty($addon['name'])) {
                    $product->addons()->create([
                        'name' => $addon['name'],
                        'price' => $addon['price'] ?? 0
                    ]);
                }
            }
        }

        return response()->json(['message' => 'Product updated successfully', 'updated_fields' => array_keys($updateData)]);
    }

    public function destroy(Request $request, $id)
    {
        $restaurant = $request->user()->restaurants()->first();
        
        if (!$restaurant) {
            return response()->json(['message' => 'No restaurant found'], 404);
        }

        $restaurant->products()->detach($id);

        return response()->json(['message' => 'Product removed from your menu']);
    }
}
