<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;

class CategoryManagementController extends Controller
{
    public function index()
    {
        $categories = Category::with(['restaurants'])
            ->latest()
            ->paginate(20);

        return response()->json($categories);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $category = Category::create($validated);

        return response()->json($category, 201);
    }

    public function show(Category $category)
    {
        return response()->json($category->load(['restaurants', 'products']));
    }

    public function update(Request $request, Category $category)
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $category->update($validated);

        return response()->json($category);
    }

    public function destroy(Category $category)
    {
        $category->delete();
        return response()->json(null, 204);
    }

    public function toggleStatus(Category $category)
    {
        $category->update(['is_active' => !$category->is_active]);
        return response()->json($category);
    }

    public function assignToRestaurant(Request $request, Category $category)
    {
        $validated = $request->validate([
            'restaurant_id' => 'required|exists:restaurants,id',
        ]);

        $category->restaurants()->attach($validated['restaurant_id']);

        return response()->json($category->load('restaurants'));
    }

    public function removeFromRestaurant(Request $request, Category $category)
    {
        $validated = $request->validate([
            'restaurant_id' => 'required|exists:restaurants,id',
        ]);

        $category->restaurants()->detach($validated['restaurant_id']);

        return response()->json($category->load('restaurants'));
    }

    public function restaurants(Category $category)
    {
        $restaurants = $category->restaurants()->paginate(20);
        return response()->json($restaurants);
    }

    public function reorder(Request $request)
    {
        $validated = $request->validate([
            'categories' => 'required|array',
            'categories.*.id' => 'required|exists:categories,id',
            'categories.*.order' => 'required|integer|min:0',
        ]);

        foreach ($validated['categories'] as $category) {
            Category::where('id', $category['id'])->update(['order' => $category['order']]);
        }

        return response()->json(['message' => 'Categories reordered successfully']);
    }
}