<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\CommissionRule;
use App\Models\Payout;
use App\Models\Restaurant;
use App\Models\DeliveryBoy;
use App\Models\Order;
use App\Models\Wallet;
use App\Http\Resources\PaymentResource;
use App\Http\Resources\CommissionRuleResource;
use App\Http\Resources\PayoutResource;
use App\Http\Requests\Admin\CreateCommissionRuleRequest;
use App\Http\Requests\Admin\UpdateCommissionRuleRequest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class PaymentCommissionManagementController extends Controller
{
    /**
     * Get all payments with filters.
     */
    public function payments(Request $request): JsonResponse
    {
        $query = Payment::with(['order', 'user', 'restaurant'])
            ->when($request->search, function($q) use ($request) {
                $q->where(function($query) use ($request) {
                    $query->where('transaction_id', 'like', "%{$request->search}%")
                          ->orWhere('payment_method', 'like', "%{$request->search}%");
                });
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($request->payment_method, function($q) use ($request) {
                $q->where('payment_method', $request->payment_method);
            })
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->when($request->min_amount, function($q) use ($request) {
                $q->where('amount', '>=', $request->min_amount);
            })
            ->when($request->max_amount, function($q) use ($request) {
                $q->where('amount', '<=', $request->max_amount);
            })
            ->when($request->sort_by, function($q) use ($request) {
                $q->orderBy($request->sort_by, $request->sort_order ?? 'desc');
            }, function($q) {
                $q->orderBy('created_at', 'desc');
            });

        $payments = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => PaymentResource::collection($payments),
            'meta' => [
                'current_page' => $payments->currentPage(),
                'last_page' => $payments->lastPage(),
                'per_page' => $payments->perPage(),
                'total' => $payments->total(),
            ]
        ]);
    }

    /**
     * Get payment details.
     */
    public function paymentDetails(Payment $payment): JsonResponse
    {
        $payment->load(['order', 'user', 'restaurant', 'refunds']);

        return response()->json([
            'success' => true,
            'data' => new PaymentResource($payment)
        ]);
    }

    /**
     * Get commission rules.
     */
    public function commissionRules(Request $request): JsonResponse
    {
        $query = CommissionRule::with(['commissionable'])
            ->when($request->type, function($q) use ($request) {
                $q->where('type', $request->type);
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($request->commissionable_type, function($q) use ($request) {
                $q->where('commissionable_type', $request->commissionable_type);
            })
            ->when($request->sort_by, function($q) use ($request) {
                $q->orderBy($request->sort_by, $request->sort_order ?? 'desc');
            }, function($q) {
                $q->orderBy('created_at', 'desc');
            });

        $rules = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => CommissionRuleResource::collection($rules),
            'meta' => [
                'current_page' => $rules->currentPage(),
                'last_page' => $rules->lastPage(),
                'per_page' => $rules->perPage(),
                'total' => $rules->total(),
            ]
        ]);
    }

    /**
     * Create commission rule.
     */
    public function createCommissionRule(CreateCommissionRuleRequest $request): JsonResponse
    {
        $rule = CommissionRule::create($request->validated());

        return response()->json([
            'success' => true,
            'message' => 'Commission rule created successfully',
            'data' => new CommissionRuleResource($rule->load(['commissionable']))
        ], 201);
    }

    /**
     * Update commission rule.
     */
    public function updateCommissionRule(UpdateCommissionRuleRequest $request, CommissionRule $rule): JsonResponse
    {
        $rule->update($request->validated());

        return response()->json([
            'success' => true,
            'message' => 'Commission rule updated successfully',
            'data' => new CommissionRuleResource($rule->fresh(['commissionable']))
        ]);
    }

    /**
     * Delete commission rule.
     */
    public function deleteCommissionRule(CommissionRule $rule): JsonResponse
    {
        $rule->delete();

        return response()->json([
            'success' => true,
            'message' => 'Commission rule deleted successfully'
        ]);
    }

    /**
     * Get restaurant commissions.
     */
    public function restaurantCommissions(Restaurant $restaurant, Request $request): JsonResponse
    {
        $query = $restaurant->commissions()
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->orderBy('created_at', 'desc');

        $commissions = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $commissions
        ]);
    }

    /**
     * Get delivery boy commissions.
     */
    public function deliveryBoyCommissions(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $query = $deliveryBoy->commissions()
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->orderBy('created_at', 'desc');

        $commissions = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $commissions
        ]);
    }

    /**
     * Get payouts.
     */
    public function payouts(Request $request): JsonResponse
    {
        $query = Payout::with(['user', 'payable'])
            ->when($request->type, function($q) use ($request) {
                $q->where('type', $request->type);
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->when($request->payable_type, function($q) use ($request) {
                $q->where('payable_type', $request->payable_type);
            })
            ->when($request->sort_by, function($q) use ($request) {
                $q->orderBy($request->sort_by, $request->sort_order ?? 'desc');
            }, function($q) {
                $q->orderBy('created_at', 'desc');
            });

        $payouts = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => PayoutResource::collection($payouts),
            'meta' => [
                'current_page' => $payouts->currentPage(),
                'last_page' => $payouts->lastPage(),
                'per_page' => $payouts->perPage(),
                'total' => $payouts->total(),
            ]
        ]);
    }

    /**
     * Process restaurant payout.
     */
    public function processRestaurantPayout(Restaurant $restaurant, Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'payment_method' => 'required|string|max:50',
            'notes' => 'nullable|string|max:500'
        ]);

        return DB::transaction(function () use ($request, $restaurant) {
            // Create payout record
            $payout = Payout::create([
                'user_id' => $restaurant->user_id,
                'payable_id' => $restaurant->id,
                'payable_type' => Restaurant::class,
                'amount' => $request->amount,
                'type' => 'restaurant_settlement',
                'status' => 'processing',
                'payment_method' => $request->payment_method,
                'notes' => $request->notes
            ]);

            // Process actual payment (payment gateway integration would go here)
            // For now, we'll mark as paid immediately
            $payout->update([
                'status' => 'paid',
                'paid_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Restaurant payout processed successfully',
                'data' => new PayoutResource($payout->load(['user', 'payable']))
            ]);
        });
    }

    /**
     * Process delivery boy payout.
     */
    public function processDeliveryBoyPayout(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $deliveryBoy->total_earnings,
            'payment_method' => 'required|string|max:50',
            'notes' => 'nullable|string|max:500'
        ]);

        return DB::transaction(function () use ($request, $deliveryBoy) {
            // Create payout record
            $payout = Payout::create([
                'user_id' => $deliveryBoy->user_id,
                'payable_id' => $deliveryBoy->id,
                'payable_type' => DeliveryBoy::class,
                'amount' => $request->amount,
                'type' => 'delivery_commission',
                'status' => 'processing',
                'payment_method' => $request->payment_method,
                'notes' => $request->notes
            ]);

            // Update delivery boy earnings
            $deliveryBoy->decrement('total_earnings', $request->amount);

            // Process actual payment (payment gateway integration would go here)
            // For now, we'll mark as paid immediately
            $payout->update([
                'status' => 'paid',
                'paid_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy payout processed successfully',
                'data' => new PayoutResource($payout->load(['user', 'payable']))
            ]);
        });
    }

    /**
     * Get payment analytics.
     */
    public function paymentAnalytics(Request $request): JsonResponse
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $analytics = [
            'total_payments' => Payment::whereBetween('created_at', [$startDate, $endDate])->count(),
            'total_amount' => Payment::whereBetween('created_at', [$startDate, $endDate])->sum('amount'),
            'successful_payments' => Payment::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'completed')->count(),
            'failed_payments' => Payment::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'failed')->count(),
            'pending_payments' => Payment::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'pending')->count(),
        ];

        // Payment method breakdown
        $methodBreakdown = Payment::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', 'completed')
            ->selectRaw('payment_method, COUNT(*) as count, SUM(amount) as total')
            ->groupBy('payment_method')
            ->get();

        $analytics['payment_method_breakdown'] = $methodBreakdown;

        // Daily payments
        $dailyPayments = Payment::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', 'completed')
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count, SUM(amount) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $analytics['daily_payments'] = $dailyPayments;

        return response()->json([
            'success' => true,
            'data' => $analytics
        ]);
    }

    /**
     * Get commission analytics.
     */
    public function commissionAnalytics(Request $request): JsonResponse
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $analytics = [
            'total_commissions' => DB::table('commissions')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount'),
            'restaurant_commissions' => DB::table('commissions')
                ->where('commissionable_type', Restaurant::class)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount'),
            'delivery_boy_commissions' => DB::table('commissions')
                ->where('commissionable_type', DeliveryBoy::class)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount'),
        ];

        // Commission breakdown by type
        $typeBreakdown = DB::table('commissions')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('type, SUM(amount) as total')
            ->groupBy('type')
            ->get();

        $analytics['commission_type_breakdown'] = $typeBreakdown;

        // Daily commissions
        $dailyCommissions = DB::table('commissions')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, SUM(amount) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $analytics['daily_commissions'] = $dailyCommissions;

        return response()->json([
            'success' => true,
            'data' => $analytics
        ]);
    }

    /**
     * Get payout analytics.
     */
    public function payoutAnalytics(Request $request): JsonResponse
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $analytics = [
            'total_payouts' => Payout::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'paid')->sum('amount'),
            'restaurant_payouts' => Payout::where('payable_type', Restaurant::class)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'paid')->sum('amount'),
            'delivery_boy_payouts' => Payout::where('payable_type', DeliveryBoy::class)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'paid')->sum('amount'),
            'pending_payouts' => Payout::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'processing')->sum('amount'),
        ];

        // Payout breakdown by type
        $typeBreakdown = Payout::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', 'paid')
            ->selectRaw('type, SUM(amount) as total, COUNT(*) as count')
            ->groupBy('type')
            ->get();

        $analytics['payout_type_breakdown'] = $typeBreakdown;

        // Daily payouts
        $dailyPayouts = Payout::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', 'paid')
            ->selectRaw('DATE(created_at) as date, SUM(amount) as total, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $analytics['daily_payouts'] = $dailyPayouts;

        return response()->json([
            'success' => true,
            'data' => $analytics
        ]);
    }

    /**
     * Get wallet transactions.
     */
    public function walletTransactions(Request $request): JsonResponse
    {
        $query = Wallet::with(['user'])
            ->when($request->type, function($q) use ($request) {
                $q->where('type', $request->type);
            })
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->orderBy('created_at', 'desc');

        $transactions = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ]);
    }

    /**
     * Export payment data.
     */
    public function exportPayments(Request $request): JsonResponse
    {
        $query = Payment::with(['order', 'user', 'restaurant'])
            ->when($request->date_from, function($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            })
            ->when($request->date_to, function($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            });

        $payments = $query->get();

        // Generate CSV data
        $csvData = $this->generatePaymentCsv($payments);

        return response()->json([
            'success' => true,
            'data' => $csvData
        ]);
    }

    /**
     * Generate CSV data for payments.
     */
    private function generatePaymentCsv($payments): array
    {
        $headers = [
            'Payment ID',
            'Transaction ID',
            'Order ID',
            'User',
            'Restaurant',
            'Amount',
            'Payment Method',
            'Status',
            'Created At',
            'Updated At'
        ];

        $data = [$headers];

        foreach ($payments as $payment) {
            $data[] = [
                $payment->id,
                $payment->transaction_id,
                $payment->order_id,
                $payment->user->name ?? 'N/A',
                $payment->restaurant->name ?? 'N/A',
                $payment->amount,
                $payment->payment_method,
                $payment->status,
                $payment->created_at->format('Y-m-d H:i:s'),
                $payment->updated_at->format('Y-m-d H:i:s')
            ];
        }

        return $data;
    }
}