<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\{User, Restaurant, Order, DeliveryBoy, Product, Payment};
use Illuminate\Http\{Request, JsonResponse};
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class SuperAdminDashboardController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $period = $request->get('period', '30'); // days
        $startDate = Carbon::now()->subDays($period);

        return response()->json([
            'success' => true,
            'data' => [
                'stats' => $this->getStats(),
                'charts' => $this->getCharts($startDate),
                'recent_activities' => $this->getRecentActivities(),
                'top_performers' => $this->getTopPerformers($startDate),
            ]
        ]);
    }

    private function getStats(): array
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();

        $todayRevenue = Order::whereDate('created_at', $today)->sum('total') ?? 0;
        $yesterdayRevenue = Order::whereDate('created_at', $yesterday)->sum('total') ?? 0;
        $todayOrders = Order::whereDate('created_at', $today)->count();
        $yesterdayOrders = Order::whereDate('created_at', $yesterday)->count();

        return [
            'total_revenue' => [
                'value' => Order::sum('total') ?? 0,
                'today' => $todayRevenue,
                'growth' => $this->calculateGrowth($todayRevenue, $yesterdayRevenue)
            ],
            'total_orders' => [
                'value' => Order::count(),
                'today' => $todayOrders,
                'pending' => Order::where('status', 'pending')->count(),
                'growth' => $this->calculateGrowth($todayOrders, $yesterdayOrders)
            ],
            'total_restaurants' => [
                'value' => Restaurant::count(),
                'active' => Restaurant::where('status', 'approved')->count(),
                'pending' => Restaurant::where('status', 'pending')->count(),
            ],
            'total_users' => [
                'value' => User::count(),
                'today' => User::whereDate('created_at', $today)->count(),
                'verified' => User::whereNotNull('email_verified_at')->count(),
            ],
            'delivery_boys' => [
                'total' => DeliveryBoy::count(),
                'active_now' => DeliveryBoy::whereIn('status', ['online', 'on_delivery'])->count(),
                'online' => DeliveryBoy::where('status', 'online')->count(),
                'offline' => DeliveryBoy::where('status', 'offline')->count(),
                'on_delivery' => DeliveryBoy::where('status', 'on_delivery')->count(),
                'blocked' => DeliveryBoy::where('status', 'blocked')->count(),
                'verified' => DeliveryBoy::where('verification_status', 'verified')->count(),
                'pending_verification' => DeliveryBoy::where('verification_status', 'pending')->count(),
                'top_earner' => DeliveryBoy::orderByDesc('wallet_balance')->first(),
                'best_rated' => DeliveryBoy::where('total_ratings', '>', 0)->orderByDesc('rating')->first(),
            ],
            'live_stats' => [
                'active_orders' => Order::whereIn('status', ['accepted', 'preparing', 'ready', 'picked_up'])->count(),
                'pending_approvals' => Restaurant::where('status', 'pending')->count(),
            ]
        ];
    }

    private function getCharts($startDate): array
    {
        return [
            'revenue' => Order::where('created_at', '>=', $startDate)
                ->selectRaw('DATE(created_at) as date, SUM(total) as amount')
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
            'orders' => Order::where('created_at', '>=', $startDate)
                ->selectRaw('DATE(created_at) as date, COUNT(*) as count, status')
                ->groupBy('date', 'status')
                ->orderBy('date')
                ->get(),
            'order_status' => Order::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get(),
            'payment_methods' => Payment::selectRaw('payment_method, COUNT(*) as count, SUM(amount) as total')
                ->groupBy('payment_method')
                ->get(),
        ];
    }

    private function getRecentActivities(): array
    {
        $orders = Order::with('user', 'restaurant')
            ->latest()
            ->limit(5)
            ->get()
            ->map(fn($o) => [
                'type' => 'order',
                'title' => "New Order #{$o->order_number}",
                'description' => ($o->user ? $o->user->name : 'Guest') . " ordered from " . ($o->restaurant ? $o->restaurant->name : 'Restaurant'),
                'amount' => $o->total,
                'time' => $o->created_at->diffForHumans(),
            ]);

        $restaurants = Restaurant::latest()
            ->limit(3)
            ->get()
            ->map(fn($r) => [
                'type' => 'restaurant',
                'title' => 'New Restaurant',
                'description' => "{$r->name} registered",
                'time' => $r->created_at->diffForHumans(),
            ]);

        $users = User::latest()
            ->limit(3)
            ->get()
            ->map(fn($u) => [
                'type' => 'user',
                'title' => 'New User',
                'description' => "{$u->name} joined",
                'time' => $u->created_at->diffForHumans(),
            ]);

        return collect($orders)->merge($restaurants)->merge($users)->take(10)->values()->all();
    }

    private function getTopPerformers($startDate): array
    {
        return [
            'restaurants' => Restaurant::withCount(['orders' => fn($q) => $q->where('created_at', '>=', $startDate)])
                ->withSum(['orders as revenue' => fn($q) => $q->where('created_at', '>=', $startDate)], 'total')
                ->orderByDesc('revenue')
                ->limit(5)
                ->get(),
            'products' => Product::withCount(['orderItems' => fn($q) => $q->whereHas('order', fn($q2) => $q2->where('created_at', '>=', $startDate))])
                ->orderByDesc('order_items_count')
                ->limit(5)
                ->get(),
            'delivery_boys' => DeliveryBoy::with('user')
                ->withCount(['deliveries' => fn($q) => $q->where('created_at', '>=', $startDate)])
                ->orderByDesc('total_deliveries')
                ->limit(5)
                ->get(),
            'top_earning_delivery_boys' => DeliveryBoy::with('user')
                ->orderByDesc('wallet_balance')
                ->limit(5)
                ->get(),
            'best_rated_delivery_boys' => DeliveryBoy::with('user')
                ->where('total_ratings', '>', 0)
                ->orderByDesc('rating')
                ->limit(5)
                ->get(),
        ];
    }

    private function calculateGrowth($current, $previous): float
    {
        if ($previous == 0) return $current > 0 ? 100 : 0;
        return round((($current - $previous) / $previous) * 100, 2);
    }
}
