<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\Restaurant;
use App\Models\RestaurantPayment;
use App\Models\CommissionRule;
use App\Http\Resources\RestaurantResource;
use App\Http\Requests\Admin\UpdateRestaurantRequest;
use App\Services\RestaurantService;
use App\Repositories\RestaurantRepository;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class RestaurantManagementController extends Controller
{
    protected $restaurantService;
    protected $restaurantRepository;

    public function __construct(RestaurantService $restaurantService, RestaurantRepository $restaurantRepository)
    {
        $this->restaurantService = $restaurantService;
        $this->restaurantRepository = $restaurantRepository;
    }
    /**
     * Display a listing of all restaurants.
     */
    public function index(Request $request): JsonResponse
    {
        $filters = [
            'status' => $request->status,
            'search' => $request->search,
            'sort_by' => $request->sort_by,
            'sort_order' => $request->sort_order,
            'per_page' => $request->per_page ?? 20
        ];

        $restaurants = $this->restaurantRepository->search($filters);

        return response()->json([
            'success' => true,
            'data' => RestaurantResource::collection($restaurants),
            'meta' => [
                'current_page' => $restaurants->currentPage(),
                'last_page' => $restaurants->lastPage(),
                'per_page' => $restaurants->perPage(),
                'total' => $restaurants->total(),
            ]
        ]);
    }

    /**
     * Display pending restaurants for approval.
     */
    public function pending(Request $request): JsonResponse
    {
        $restaurants = $this->restaurantRepository->getPendingApproval($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => RestaurantResource::collection($restaurants),
            'meta' => [
                'current_page' => $restaurants->currentPage(),
                'last_page' => $restaurants->lastPage(),
                'per_page' => $restaurants->perPage(),
                'total' => $restaurants->total(),
            ]
        ]);
    }

    /**
     * Store a newly created restaurant.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'required|string',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|unique:restaurants,email',
            'description' => 'nullable|string',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'monthly_fee' => 'nullable|numeric|min:0',
            'commission_rate' => 'nullable|numeric|min:0|max:100',
            'create_owner' => 'nullable|boolean',
            'owner_name' => 'required_if:create_owner,true|string',
            'owner_email' => 'required_if:create_owner,true|email|unique:users,email',
            'owner_phone' => 'nullable|string',
            'owner_password' => 'required_if:create_owner,true|string|min:8',
        ]);

        $userId = null;

        // Create owner account if requested
        if ($request->create_owner && $request->owner_name && $request->owner_email && $request->owner_password) {
            $owner = \App\Models\User::create([
                'name' => $request->owner_name,
                'email' => $request->owner_email,
                'phone' => $request->owner_phone,
                'password' => bcrypt($request->owner_password),
                'email_verified_at' => now()
            ]);
            
            // Assign manager role
            $owner->assignRole('manager');
            $userId = $owner->id;
        }

        $restaurant = Restaurant::create([
            'user_id' => $userId,
            'name' => $validated['name'],
            'address' => $validated['address'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'description' => $validated['description'] ?? null,
            'latitude' => !empty($validated['latitude']) ? $validated['latitude'] : 0,
            'longitude' => !empty($validated['longitude']) ? $validated['longitude'] : 0,
            'monthly_fee' => $validated['monthly_fee'] ?? 0,
            'commission_rate' => $validated['commission_rate'] ?? 0,
            'status' => 'approved',
            'operational_status' => 'open'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant created successfully',
            'data' => new RestaurantResource($restaurant)
        ], 201);
    }

    /**
     * Display the specified restaurant.
     */
    public function show(Restaurant $restaurant): JsonResponse
    {
        $restaurant = $this->restaurantRepository->findById($restaurant->id);

        return response()->json([
            'success' => true,
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Update the specified restaurant.
     */
    public function update(UpdateRestaurantRequest $request, Restaurant $restaurant): JsonResponse
    {
        $validated = $request->validated();
        
        // Update owner details if provided
        if ($restaurant->user_id && ($request->has('owner_name') || $request->has('owner_email') || $request->has('owner_phone'))) {
            $ownerData = [];
            if ($request->filled('owner_name')) $ownerData['name'] = $request->owner_name;
            if ($request->filled('owner_email')) $ownerData['email'] = $request->owner_email;
            if ($request->filled('owner_phone')) $ownerData['phone'] = $request->owner_phone;
            
            if (!empty($ownerData)) {
                $restaurant->user()->update($ownerData);
            }
        }
        
        $restaurant = $this->restaurantService->updateRestaurant($restaurant, $validated);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant updated successfully',
            'data' => new RestaurantResource($restaurant->load('user'))
        ]);
    }

    /**
     * Approve restaurant registration.
     */
    public function approve(Restaurant $restaurant): JsonResponse
    {
        $restaurant = $this->restaurantService->approveRestaurant($restaurant);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant approved successfully',
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Reject restaurant registration.
     */
    public function reject(Restaurant $restaurant, Request $request): JsonResponse
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        $restaurant = $this->restaurantService->rejectRestaurant($restaurant, $request->reason);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant rejected successfully',
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Suspend restaurant.
     */
    public function suspend(Restaurant $restaurant, Request $request): JsonResponse
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        $restaurant = $this->restaurantService->suspendRestaurant($restaurant, $request->reason);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant suspended successfully',
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Activate suspended restaurant.
     */
    public function activate(Restaurant $restaurant): JsonResponse
    {
        $restaurant = $this->restaurantService->activateRestaurant($restaurant);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant activated successfully',
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Force open/close restaurant.
     */
    public function updateOperationalStatus(Restaurant $restaurant, Request $request): JsonResponse
    {
        $request->validate([
            'operational_status' => 'required|in:open,closed,busy'
        ]);

        $restaurant = $this->restaurantService->updateOperationalStatus($restaurant, $request->operational_status);

        return response()->json([
            'success' => true,
            'message' => 'Restaurant operational status updated successfully',
            'data' => new RestaurantResource($restaurant)
        ]);
    }

    /**
     * Get restaurant sales analytics.
     */
    public function analytics(Restaurant $restaurant, Request $request): JsonResponse
    {
        $analytics = $this->restaurantService->getRestaurantAnalytics($restaurant, $request->all());

        return response()->json([
            'success' => true,
            'data' => $analytics
        ]);
    }

    /**
     * Upload restaurant banners and logos.
     */
    public function uploadMedia(Restaurant $restaurant, Request $request): JsonResponse
    {
        $request->validate([
            'type' => 'required|in:logo,cover,banner',
            'image' => 'required|image|max:5120' // 5MB max
        ]);

        $path = $this->restaurantService->uploadMedia($restaurant, $request->type, $request->file('image'));

        return response()->json([
            'success' => true,
            'message' => 'Media uploaded successfully',
            'data' => [
                'path' => $path,
                'url' => Storage::url($path)
            ]
        ]);
    }

    /**
     * Get restaurant payment history.
     */
    public function paymentHistory(Restaurant $restaurant): JsonResponse
    {
        $payments = $restaurant->payments()->with('recordedBy:id,name')->get();

        return response()->json([
            'success' => true,
            'data' => $payments
        ]);
    }

    /**
     * Record restaurant payment.
     */
    public function recordPayment(Restaurant $restaurant, Request $request): JsonResponse
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,card,other',
            'transaction_id' => 'nullable|string',
            'notes' => 'nullable|string'
        ]);

        // Calculate period
        $paymentDate = \Carbon\Carbon::parse($validated['payment_date']);
        $periodStart = $paymentDate->copy()->startOfMonth();
        $periodEnd = $paymentDate->copy()->endOfMonth();

        $payment = RestaurantPayment::create([
            'restaurant_id' => $restaurant->id,
            'amount' => $validated['amount'],
            'payment_date' => $validated['payment_date'],
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'status' => 'paid',
            'payment_method' => $validated['payment_method'],
            'transaction_id' => $validated['transaction_id'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'recorded_by' => $request->user()->id
        ]);

        // Update restaurant payment dates
        $restaurant->update([
            'last_payment_date' => $validated['payment_date'],
            'next_payment_date' => $paymentDate->copy()->addMonth(),
            'payment_status' => 'paid'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Payment recorded successfully',
            'data' => $payment
        ]);
    }

    /**
     * Update restaurant monthly fee.
     */
    public function updateFee(Restaurant $restaurant, Request $request): JsonResponse
    {
        $validated = $request->validate([
            'monthly_fee' => 'required|numeric|min:0'
        ]);

        $restaurant->update(['monthly_fee' => $validated['monthly_fee']]);

        return response()->json([
            'success' => true,
            'message' => 'Monthly fee updated successfully',
            'data' => $restaurant
        ]);
    }

    /**
     * Delete restaurant.
     */
    public function destroy(Restaurant $restaurant): JsonResponse
    {
        $restaurant->delete();

        return response()->json([
            'success' => true,
            'message' => 'Restaurant deleted successfully'
        ]);
    }
}