<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\GlobalProduct;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class GlobalProductController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = GlobalProduct::with('category');
            
            if ($request->category_id) {
                $query->where('category_id', $request->category_id);
            }
            
            if ($request->search) {
                $query->where('name', 'like', '%' . $request->search . '%');
            }
            
            $products = $query->latest()->paginate($request->per_page ?? 20);
            
            return response()->json(['success' => true, 'data' => $products]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'category_id' => 'required|exists:categories,id',
                'name' => 'required|string|max:255|unique:global_products,name',
                'description' => 'nullable|string',
                'image' => 'nullable|image|max:2048'
            ]);
            
            $data = $request->except('image');
            
            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('global_products', 'public');
            }
            
            $product = GlobalProduct::create($data);
            
            return response()->json(['success' => true, 'message' => 'Global product created', 'data' => $product->load('category')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $product = GlobalProduct::findOrFail($id);
            
            $request->validate([
                'category_id' => 'required|exists:categories,id',
                'name' => 'required|string|max:255|unique:global_products,name,' . $id,
                'description' => 'nullable|string',
                'image' => 'nullable|image|max:2048'
            ]);
            
            $data = $request->except('image');
            
            if ($request->hasFile('image')) {
                if ($product->image) Storage::disk('public')->delete($product->image);
                $data['image'] = $request->file('image')->store('global_products', 'public');
            }
            
            $product->update($data);
            
            return response()->json(['success' => true, 'message' => 'Global product updated', 'data' => $product->load('category')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $product = GlobalProduct::findOrFail($id);
            if ($product->image) Storage::disk('public')->delete($product->image);
            $product->delete();
            
            return response()->json(['success' => true, 'message' => 'Global product deleted']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function toggleActive($id)
    {
        try {
            $product = GlobalProduct::findOrFail($id);
            $product->update(['is_active' => !$product->is_active]);
            
            return response()->json(['success' => true, 'message' => 'Status updated']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function assignToRestaurant(Request $request, $id)
    {
        try {
            $request->validate([
                'restaurant_id' => 'required|exists:restaurants,id',
                'price' => 'required|numeric|min:0',
                'offer_price' => 'nullable|numeric|min:0',
                'discount_percentage' => 'nullable|numeric|min:0|max:100',
                'special_notes' => 'nullable|string',
                'available_from' => 'nullable|date_format:H:i',
                'available_until' => 'nullable|date_format:H:i',
                'min_order_quantity' => 'nullable|integer|min:1'
            ]);

            $globalProduct = GlobalProduct::findOrFail($id);
            $restaurant = \App\Models\Restaurant::findOrFail($request->restaurant_id);

            // Check if already assigned
            if ($restaurant->products()->where('product_id', $globalProduct->id)->exists()) {
                return response()->json(['success' => false, 'message' => 'Product already assigned to this restaurant'], 400);
            }

            // Assign product to restaurant
            $restaurant->products()->attach($globalProduct->id, [
                'price' => $request->price,
                'offer_price' => $request->offer_price,
                'discount_percentage' => $request->discount_percentage,
                'special_notes' => $request->special_notes,
                'available_from' => $request->available_from,
                'available_until' => $request->available_until,
                'min_order_quantity' => $request->min_order_quantity ?? 1,
                'is_available' => 1
            ]);

            return response()->json(['success' => true, 'message' => 'Product assigned to restaurant successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
}
