<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeliveryBoy;
use App\Models\DeliveryBoyDocument;
use App\Models\DeliveryBoyArea;
use App\Models\DeliveryBoyVehicle;
use App\Models\User;
use App\Http\Requests\Admin\StoreDeliveryBoyRequest;
use App\Http\Requests\Admin\UpdateDeliveryBoyRequest;
use App\Http\Requests\Admin\VerifyDeliveryDocumentRequest;
use App\Services\ActivityLoggerService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\{Hash, DB, Log};

class DeliveryBoyController extends Controller
{
    protected $activityLogger;

    public function __construct(ActivityLoggerService $activityLogger)
    {
        $this->activityLogger = $activityLogger;
    }

    public function getDashboardStats(): JsonResponse
    {
        $stats = [
            'total_delivery_boys' => DeliveryBoy::count(),
            'online_count' => DeliveryBoy::where('status', 'online')->count(),
            'offline_count' => DeliveryBoy::where('status', 'offline')->count(),
            'on_delivery_count' => DeliveryBoy::where('status', 'on_delivery')->count(),
            'blocked_count' => DeliveryBoy::where('status', 'blocked')->count(),
            'pending_verification' => DeliveryBoy::where('verification_status', 'pending')->count(),
            'verified_count' => DeliveryBoy::where('verification_status', 'verified')->count(),
            'avg_rating' => DeliveryBoy::avg('rating') ?? 0,
            'total_deliveries_today' => DeliveryBoy::whereHas('deliveries', function($q) {
                $q->whereDate('created_at', today());
            })->count()
        ];

        return response()->json([
            'success' => true,
            'data' => ['stats' => $stats]
        ]);
    }

    public function index(Request $request): JsonResponse
    {
        // Skip authorization for manager route
        if (!request()->is('api/manager/*')) {
            $this->authorize('viewAny', DeliveryBoy::class);
        }

        $query = DeliveryBoy::select('id', 'user_id', 'full_name', 'phone', 'email', 'city', 'status', 'verification_status', 'rating', 'total_deliveries', 'wallet_balance', 'commission_rate', 'joined_date')
            ->with(['user:id,name', 'vehicles:id,delivery_boy_id,vehicle_type']);

        // Filters
        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->verification_status) {
            $query->where('verification_status', $request->verification_status);
        }

        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('full_name', 'like', "%{$request->search}%")
                  ->orWhere('phone', 'like', "%{$request->search}%")
                  ->orWhere('email', 'like', "%{$request->search}%");
            });
        }

        $deliveryBoys = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $deliveryBoys
        ]);
    }

    public function show(DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('view', $deliveryBoy);

        $deliveryBoy->load(['user', 'vehicles', 'documents', 'areas', 'earnings', 'ratings']);

        return response()->json([
            'success' => true,
            'data' => $deliveryBoy
        ]);
    }

    public function store(StoreDeliveryBoyRequest $request): JsonResponse
    {
        $this->authorize('create', DeliveryBoy::class);

        DB::beginTransaction();
        try {
            $validated = $request->validated();

            $user = User::create([
                'name' => $validated['full_name'],
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'password' => Hash::make($validated['password']),
                'email_verified_at' => now()
            ]);

            $user->assignRole('delivery-boy');

            $deliveryBoy = DeliveryBoy::create([
                'user_id' => $user->id,
                'full_name' => $validated['full_name'],
                'phone' => $validated['phone'],
                'email' => $validated['email'],
                'address' => $validated['address'] ?? null,
                'city' => $validated['city'] ?? null,
                'postal_code' => $validated['postal_code'] ?? null,
                'latitude' => $validated['latitude'] ?? null,
                'longitude' => $validated['longitude'] ?? null,
                'national_id' => $validated['national_id'] ?? null,
                'date_of_birth' => $validated['date_of_birth'] ?? null,
                'emergency_contact_name' => $validated['emergency_contact_name'] ?? null,
                'emergency_contact_phone' => $validated['emergency_contact_phone'] ?? null,
                'commission_rate' => $validated['commission_rate'] ?? 10.00,
                'joined_date' => now(),
                'status' => 'offline',
                'verification_status' => 'pending'
            ]);

            $this->activityLogger->deliveryBoyCreated($deliveryBoy);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy created successfully',
                'data' => $deliveryBoy->load('user')
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create delivery boy', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create delivery boy',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(UpdateDeliveryBoyRequest $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            $validated = $request->validated();
            $changes = array_diff_assoc($validated, $deliveryBoy->toArray());

            $deliveryBoy->update($validated);

            if (!empty($changes)) {
                $this->activityLogger->deliveryBoyUpdated($deliveryBoy, $changes);
            }

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy updated successfully',
                'data' => $deliveryBoy->load('user')
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to update delivery boy', ['id' => $deliveryBoy->id, 'error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update delivery boy'
            ], 500);
        }
    }

    public function block(Request $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('block', $deliveryBoy);

        try {
            $action = $deliveryBoy->status === 'blocked' ? 'offline' : 'blocked';
            $deliveryBoy->update(['status' => $action]);

            if ($action === 'blocked') {
                $this->activityLogger->deliveryBoyBlocked($deliveryBoy, $request->reason);
            } else {
                $this->activityLogger->deliveryBoyUnblocked($deliveryBoy);
            }

            return response()->json([
                'success' => true,
                'message' => $action === 'blocked' ? 'Delivery boy blocked successfully' : 'Delivery boy unblocked successfully',
                'data' => $deliveryBoy
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to block/unblock delivery boy', ['id' => $deliveryBoy->id, 'error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status'
            ], 500);
        }
    }

    public function verifyDocuments(VerifyDeliveryDocumentRequest $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('verify', $deliveryBoy);

        DB::beginTransaction();
        try {
            $validated = $request->validated();

            $deliveryBoy->update([
                'verification_status' => $validated['verification_status']
            ]);

            DeliveryBoyDocument::where('delivery_boy_id', $deliveryBoy->id)
                ->where('verification_status', 'pending')
                ->update([
                    'verification_status' => $validated['verification_status'],
                    'verified_by' => $request->user()->id,
                    'verified_at' => now(),
                    'rejection_reason' => $validated['rejection_reason'] ?? null
                ]);

            if ($validated['verification_status'] === 'verified') {
                $this->activityLogger->deliveryBoyVerified($deliveryBoy);
            } else {
                $this->activityLogger->deliveryBoyRejected($deliveryBoy, $validated['rejection_reason']);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $validated['verification_status'] === 'verified' 
                    ? 'Delivery boy verified successfully' 
                    : 'Delivery boy verification rejected',
                'data' => $deliveryBoy
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to verify delivery boy', ['id' => $deliveryBoy->id, 'error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to verify documents'
            ], 500);
        }
    }

    public function assignArea(Request $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        DB::beginTransaction();
        try {
            $validated = $request->validate([
                'area_name' => 'required|string',
                'city' => 'required|string|max:100',
                'postal_code' => 'nullable|string|max:20',
                'latitude' => 'required|numeric',
                'longitude' => 'required|numeric',
                'radius_km' => 'nullable|numeric|min:1|max:50',
                'is_primary' => 'nullable|boolean'
            ]);

            if ($validated['is_primary'] ?? false) {
                DeliveryBoyArea::where('delivery_boy_id', $deliveryBoy->id)
                    ->update(['is_primary' => false]);
            }

            $area = DeliveryBoyArea::create([
                'delivery_boy_id' => $deliveryBoy->id,
                'area_name' => $validated['area_name'],
                'city' => $validated['city'],
                'postal_code' => $validated['postal_code'] ?? null,
                'latitude' => $validated['latitude'],
                'longitude' => $validated['longitude'],
                'radius_km' => $validated['radius_km'] ?? 5.00,
                'is_primary' => $validated['is_primary'] ?? false,
                'is_active' => true
            ]);

            $this->activityLogger->areaAssigned($deliveryBoy, $area);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Area assigned successfully',
                'data' => $area
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to assign area', ['id' => $deliveryBoy->id, 'error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign area'
            ], 500);
        }
    }

    public function destroy(DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('delete', $deliveryBoy);

        try {
            $deliveryBoy->delete();

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to delete delivery boy', ['id' => $deliveryBoy->id, 'error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete delivery boy'
            ], 500);
        }
    }

    public function addVehicle(Request $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            $validated = $request->validate([
                'vehicle_type' => 'required|in:bike,scooter,bicycle,car,van',
                'vehicle_number' => 'nullable|string|max:50',
                'vehicle_model' => 'nullable|string|max:100',
                'vehicle_color' => 'nullable|string|max:50',
            ]);

            $vehicle = DeliveryBoyVehicle::create([
                'delivery_boy_id' => $deliveryBoy->id,
                'vehicle_type' => $validated['vehicle_type'],
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'vehicle_model' => $validated['vehicle_model'] ?? null,
                'vehicle_color' => $validated['vehicle_color'] ?? null,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Vehicle added successfully',
                'data' => $vehicle
            ], 201);
        } catch (\Exception $e) {
            Log::error('Failed to add vehicle', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to add vehicle'
            ], 500);
        }
    }

    public function updateVehicle(Request $request, DeliveryBoy $deliveryBoy, DeliveryBoyVehicle $vehicle): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            $validated = $request->validate([
                'vehicle_type' => 'required|in:bike,scooter,bicycle,car,van',
                'vehicle_number' => 'nullable|string|max:50',
                'vehicle_model' => 'nullable|string|max:100',
                'vehicle_color' => 'nullable|string|max:50',
            ]);

            $vehicle->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Vehicle updated successfully',
                'data' => $vehicle
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to update vehicle', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update vehicle'
            ], 500);
        }
    }

    public function deleteVehicle(DeliveryBoy $deliveryBoy, DeliveryBoyVehicle $vehicle): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            $vehicle->delete();

            return response()->json([
                'success' => true,
                'message' => 'Vehicle deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to delete vehicle', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete vehicle'
            ], 500);
        }
    }

    public function uploadDocument(Request $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            $validated = $request->validate([
                'document_type' => 'required|in:national_id,driving_license,vehicle_registration,insurance,police_report',
                'document_number' => 'nullable|string|max:100',
                'file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120'
            ]);

            $filePath = $request->file('file')->store('delivery_boy_documents', 'public');

            $document = DeliveryBoyDocument::create([
                'delivery_boy_id' => $deliveryBoy->id,
                'document_type' => $validated['document_type'],
                'document_number' => $validated['document_number'] ?? null,
                'document_file' => '/storage/' . $filePath,
                'verification_status' => 'pending'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Document uploaded successfully',
                'data' => $document
            ], 201);
        } catch (\Exception $e) {
            Log::error('Failed to upload document', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload document',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function deleteDocument(DeliveryBoy $deliveryBoy, DeliveryBoyDocument $document): JsonResponse
    {
        $this->authorize('update', $deliveryBoy);

        try {
            if ($document->delivery_boy_id !== $deliveryBoy->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Document does not belong to this delivery boy'
                ], 403);
            }

            $document->delete();

            return response()->json([
                'success' => true,
                'message' => 'Document deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to delete document', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete document'
            ], 500);
        }
    }

    public function exportDeliveryBoys(Request $request)
    {
        $query = DeliveryBoy::with(['user', 'vehicles']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->verification_status) {
            $query->where('verification_status', $request->verification_status);
        }

        if ($request->delivery_boy_ids) {
            $ids = explode(',', $request->delivery_boy_ids);
            $query->whereIn('id', $ids);
        }

        $deliveryBoys = $query->get();

        $html = '<table border="1">';
        $html .= '<tr>';
        $html .= '<th>ID</th><th>Name</th><th>Phone</th><th>Email</th><th>City</th>';
        $html .= '<th>Status</th><th>Verification</th><th>Rating</th><th>Total Deliveries</th>';
        $html .= '<th>Commission Rate</th><th>Wallet Balance</th><th>Vehicle Type</th><th>Joined Date</th>';
        $html .= '</tr>';

        foreach ($deliveryBoys as $boy) {
            $html .= '<tr>';
            $html .= '<td>' . $boy->id . '</td>';
            $html .= '<td>' . $boy->full_name . '</td>';
            $html .= '<td>' . $boy->phone . '</td>';
            $html .= '<td>' . $boy->email . '</td>';
            $html .= '<td>' . ($boy->city ?? 'N/A') . '</td>';
            $html .= '<td>' . ucfirst($boy->status) . '</td>';
            $html .= '<td>' . ucfirst($boy->verification_status) . '</td>';
            $html .= '<td>' . number_format($boy->rating ?? 0, 2) . '</td>';
            $html .= '<td>' . ($boy->total_deliveries ?? 0) . '</td>';
            $html .= '<td>' . number_format($boy->commission_rate ?? 0, 2) . '%</td>';
            $html .= '<td>Rs ' . number_format($boy->wallet_balance ?? 0, 2) . '</td>';
            $html .= '<td>' . ($boy->vehicles->first()->vehicle_type ?? 'N/A') . '</td>';
            $html .= '<td>' . ($boy->joined_date ? $boy->joined_date->format('Y-m-d') : 'N/A') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</table>';

        return response($html, 200, [
            'Content-Type' => 'application/vnd.ms-excel',
            'Content-Disposition' => 'attachment; filename="delivery_boys_' . date('Y-m-d') . '.xls"'
        ]);
    }
}
