<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class CancelUndeliveredOrders extends Command
{
    protected $signature = 'orders:cancel-undelivered';
    protected $description = 'Cancel orders that are still in picked_up status at end of day';

    public function handle()
    {
        $cancelledOrders = Order::where('status', 'picked_up')
            ->whereDate('created_at', now()->toDateString())
            ->get();

        foreach ($cancelledOrders as $order) {
            $order->update([
                'status' => 'cancelled',
                'cancellation_reason' => 'Auto-cancelled: Not delivered by end of day'
            ]);

            // Create notification for manager
            $manager = User::where('restaurant_id', $order->restaurant_id)
                ->whereHas('roles', function($q) {
                    $q->where('name', 'manager');
                })->first();

            if ($manager) {
                DB::table('notifications')->insert([
                    'type' => 'App\\Notifications\\OrderCancelled',
                    'notifiable_type' => 'App\\Models\\User',
                    'notifiable_id' => $manager->id,
                    'data' => json_encode([
                        'order_id' => $order->id,
                        'order_number' => $order->order_number,
                        'message' => 'Order #' . $order->order_number . ' was auto-cancelled (not delivered by end of day)',
                        'total' => $order->total
                    ]),
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
        }

        $this->info('Cancelled ' . $cancelledOrders->count() . ' undelivered orders');
        return 0;
    }
}
