# Customer Menu Ordering System

## Overview
Complete customer-facing menu system where customers can browse products by category, select variants (sizes), choose add-ons, and add items to cart.

## Features

### 1. Restaurant Header
- Restaurant name, description, image
- Rating, delivery fee, estimated delivery time
- Clean, professional layout

### 2. Category Tabs
- Horizontal scrollable tabs
- Active category highlighted with orange gradient
- Smooth category switching
- Sticky positioning for easy navigation

### 3. Product Grid
- Card-based layout with images
- Product name, description, base price
- "Add" button for quick access
- Click anywhere on card to open details

### 4. Product Modal
- Full-screen product details
- Large product image
- Close button (X)

#### Variant Selection (Mandatory if available)
- Radio button selection
- Shows variant name and price
- Required field marked with red asterisk (*)
- Default variant pre-selected
- Orange highlight for selected variant

#### Add-ons Selection (Optional)
- Checkbox selection
- Shows addon name and price
- Multiple selection allowed
- Orange highlight for selected addons

#### Quantity Selector
- Plus/minus buttons
- Minimum quantity: 1
- Clean circular buttons

#### Total Price Calculator
- Real-time price calculation
- Base price (from variant or product)
- Add-on prices added
- Multiplied by quantity
- Large, prominent display

### 5. Add to Cart
- Validates variant selection (if required)
- Sends to backend API
- Updates cart count
- Closes modal on success

### 6. Floating Cart Button
- Shows cart item count
- Fixed bottom-right position
- Orange gradient background
- Click to navigate to cart

## User Flow

```
1. Customer visits /restaurants/{id}/menu
2. Sees restaurant header and category tabs
3. Clicks category to filter products
4. Clicks product card to open modal
5. Selects variant (if required) - MANDATORY
6. Selects add-ons (if available) - OPTIONAL
7. Adjusts quantity
8. Sees total price update in real-time
9. Clicks "Add to Cart"
10. Item added to cart, modal closes
11. Floating cart button shows item count
12. Clicks cart button to checkout
```

## API Endpoints

### Public (No Auth Required)
```
GET /api/restaurants/{id}/categories
GET /api/restaurants/{id}/products
GET /api/products/{id}/variants
GET /api/products/{id}/addons
```

### Protected (Auth Required)
```
POST /api/cart
{
  "product_id": 1,
  "variant_id": 2,      // Required if product has variants
  "addon_ids": [1, 3],  // Optional array
  "quantity": 2
}
```

## Database Structure

### Products
- `has_variants` flag determines if variant selection is required
- `base_price` used when no variants

### Variants
- Linked to product
- Has name (Half, Full, Large) and price
- `is_default` flag for pre-selection
- Only active variants shown

### Add-ons
- Linked to restaurant
- Attached to products via pivot table
- Has name and price
- Only active add-ons shown

## Validation Rules

1. **Variant Selection**
   - If `product.has_variants === true`, variant selection is MANDATORY
   - Alert shown if user tries to add without selecting variant
   - Default variant auto-selected on modal open

2. **Add-on Selection**
   - Always OPTIONAL
   - Multiple add-ons can be selected
   - No validation required

3. **Quantity**
   - Minimum: 1
   - No maximum limit
   - Affects total price calculation

## Price Calculation

```javascript
let price = 0;

// Step 1: Get base price
if (product.has_variants && selectedVariant) {
  price = variant.price;
} else {
  price = product.base_price;
}

// Step 2: Add addon prices
selectedAddons.forEach(addon => {
  price += addon.price;
});

// Step 3: Multiply by quantity
totalPrice = price * quantity;
```

## Test Data Setup

Run seeder to create test data:
```bash
php artisan db:seed --class=RestaurantMenuSeeder
```

This creates:
- Manager user (manager@test.com / password)
- Spice Garden Restaurant
- 2 Categories (Main Dishes, Drinks)
- 3 Products (Chicken Curry, Fried Rice, Coca Cola)
- Variants (Half/Full, Regular/Large)
- Add-ons (Extra Cheese, Extra Spicy)

## Testing Flow

1. Run seeder: `php artisan db:seed --class=RestaurantMenuSeeder`
2. Visit: `/restaurants/1/menu` (assuming restaurant ID is 1)
3. Click "Main Dishes" category
4. Click "Chicken Curry" product
5. Select "Full" variant (Rs 900)
6. Check "Extra Cheese" addon (+Rs 50)
7. Set quantity to 2
8. See total: Rs 1,900 (950 × 2)
9. Click "Add to Cart"
10. See cart count increase

## UI/UX Features

- **Orange Gradient Theme**: Consistent branding
- **Skeleton Loaders**: Smooth loading states
- **Hover Effects**: Interactive feedback
- **Responsive Design**: Mobile-first approach
- **Sticky Category Tabs**: Easy navigation
- **Modal Overlay**: Focus on product details
- **Real-time Updates**: Instant price calculation
- **Floating Cart**: Always accessible

## Manager Side

Managers can:
1. Login at `/manager`
2. Create categories at `/manager/categories`
3. Create products at `/manager/products`
4. Add variants (sizes) to products
5. Create add-ons at `/manager/addons`
6. Attach add-ons to products

Customers immediately see:
- All active categories
- All active products
- All active variants
- All active add-ons

---

**Complete end-to-end ordering system** 🍔🛒
